/**
 * LICENCSE + COPYRIGHT
 */
package org.vcs.bazaar.client;

import java.util.HashMap;
import java.util.Map;

import org.vcs.bazaar.client.commandline.CommandLineClientFactory;
import org.vcs.bazaar.client.core.BazaarClientException;
import org.vcs.bazaar.client.utils.BazaarUtilities;
import org.vcs.bazaar.client.xmlrpc.XmlRpcClientFactory;

/**
 * @author Guillermo Gonzalez
 *
 */
public abstract class BazaarClientFactory {

	// don't have to do null checks
	private static Map<String, BazaarClientFactory> factories = new HashMap<String, BazaarClientFactory>();
	private static BazaarClientFactory bestFactory;

	public static IBazaarClient createClient(final String clientType) {
		if (factories.containsKey(clientType)) {
			final BazaarClientFactory factory = factories.get(clientType);
			return factory != null ? factory.createClientImpl() : null;
		}
		return null;
	}

	public static final boolean isClientTypeAvailable(String clientType) {
		return factories.containsKey(clientType);
	}

	public static final void setPreferredClientType(String clientType) {
		bestFactory = factories.get(clientType);
	}

	public static final String getPreferredClientType() throws BazaarClientException {
		if (bestFactory != null) {
			return bestFactory.getClientType();
		}
		throw new BazaarClientException("No client factory found");
	}

	protected static void registerAdapterFactory(BazaarClientFactory newFactory) {
		if (newFactory != null) {
			// If a mapping for this key exist, the old value is replaced.
			factories.put(newFactory.getClientType(), newFactory);
		}
		if(bestFactory == null) {
			bestFactory = newFactory;
		}
	}

	protected abstract IBazaarClient createClientImpl();

	protected abstract String getClientType();

	public static String[] getClientTypes() {
		return factories.keySet().toArray(new String[0]);
	}

	public static String setupBestAvailableBackend(boolean force) throws BazaarClientException {
		BazaarUtilities.checkExecutable();
		String preferredClientType = null;
		try {
			// first check for the CLI client
			CommandLineClientFactory.setup(force);
			preferredClientType = CommandLineClientFactory.CLIENT_TYPE;
		} catch (BazaarClientException bce) {
			bestFactory = null;
			// if bzr isn't found, stop searching for a valid client
			// because all current impl. are based on bzr executable
			throw bce;
		}
		try {
			// now check if xmloutput plugin is available
			XmlRpcClientFactory.setup(force);
			preferredClientType = XmlRpcClientFactory.CLIENT_TYPE;
		} catch (BazaarClientException bce) {
			// TODO: inform that the fallback CLI client is used
		}
		setPreferredClientType(preferredClientType);
		return getPreferredClientType();
	}

	public static String setupBestAvailableBackend() throws BazaarClientException {
		return setupBestAvailableBackend(false);
	}

}

