/**
 * LICENSE + COPYRIGHT
 */
package org.vcs.bazaar.client.commandline.commands;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;

import org.vcs.bazaar.client.commandline.CommandLineException;
import org.vcs.bazaar.client.commandline.internal.CommandRunner;
import org.vcs.bazaar.client.commandline.syntax.ICommitOptions;
import org.vcs.bazaar.client.core.BazaarClientException;
import org.vcs.bazaar.client.utils.Assert;

/**
 * <br>
 * Represents a 'bzr commit [args]' command.<br>
 * <br>
 *
 * @author Guillermo Gonzalez
 *
 */
public class Commit extends MultiFileCommand implements ICommitOptions {

	private final String message;

	public Commit(final File workDir, final File[] files, final String message) {
		super(workDir, files);
		Assert.isTrue(message.trim().length() > 0);
		this.message = message;
		this.checkExitValue = false;

	}

	@Override
	public String getCommand() {
		return COMMAND;
	}

	@Override
	public void execute(CommandRunner runner) throws BazaarClientException {
		File tempFile = createMessageTempFile();
		try {
			setOption(ICommitOptions.FILE.with(tempFile.getPath()));
			super.execute(runner);
		} catch(CommandLineException e) {
			throw e;
		} finally {
			tempFile.delete();
		}

	}

	private File createMessageTempFile() throws CommandLineException {
		File tempFile = null;
		OutputStreamWriter osw = null;
		try {
			tempFile = File.createTempFile("bzr", "_log_message");
			osw = new OutputStreamWriter(new FileOutputStream(tempFile), "utf-8");
			osw.write(message);
		} catch (IOException e) {
			throw CommandLineException.wrapException(e);
		} finally {
			if(osw != null)
				try {
					osw.close();
				} catch (IOException e) {
					throw CommandLineException.wrapException(e);
				}
		}
		return tempFile;
	}

}
