/**
 * LICENSE + COPYRIGHT
 */
package org.vcs.bazaar.client.commandline.parser;

import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;

import org.kxml2.io.KXmlParser;
import org.vcs.bazaar.client.BazaarConflict;
import org.vcs.bazaar.client.BazaarConflictType;
import org.vcs.bazaar.client.IBazaarConflict;
import org.vcs.bazaar.client.core.BazaarClientException;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

/**
 * @author Piotr Piastucki
 */
public class XMLConflictsParser extends XMLParser {
	private static final String CONFLICTS = "conflicts";
	private static final String CONFLICT = "conflict";
	private static final String TYPE = "type";
	private final static String PATH = "path";
	private final static String MESSAGE = "message";

	public static List<IBazaarConflict> parse(String xml) throws BazaarClientException {
		KXmlParser parser = new KXmlParser();
		try {
			parser.setInput(new StringReader(xml));
			return new XMLConflictsParser().parse(parser);
		} catch (XmlPullParserException e) {
			throw BazaarClientException.wrapException(e);
		}
	}

	public List<IBazaarConflict> parse(KXmlParser aParser) throws BazaarClientException {
		List<IBazaarConflict> conflicts = new ArrayList<IBazaarConflict>();
		parser = aParser;
		try {
			parser.nextTag();
			parser.require(KXmlParser.START_TAG, null, CONFLICTS);
			int eventType = parser.getEventType();
			while (eventType != XmlPullParser.END_DOCUMENT
					&& !(eventType == XmlPullParser.END_TAG && CONFLICTS.equals(parser.getName()))) {
				if (eventType == XmlPullParser.START_TAG && CONFLICT.equals(parser.getName())) {
					IBazaarConflict conflict = parseConflict();
					if (conflict != null) {
						conflicts.add(conflict);
					}
				}
				eventType = parser.next();
			}
			return conflicts;
		} catch (XmlPullParserException e) {
			throw BazaarClientException.wrapException(e);
		} catch (IOException e) {
			throw BazaarClientException.wrapException(e);
		}
	}

	private IBazaarConflict parseConflict() throws XmlPullParserException, IOException {
		BazaarConflictType type = BazaarConflictType.UNKNOWN;
		String path = null;
		String message = null;
		int eventType = parser.next();
		while (eventType != XmlPullParser.END_DOCUMENT
				&& !(eventType == XmlPullParser.END_TAG && CONFLICT.equals(parser.getName()))) {
			if (eventType == XmlPullParser.START_TAG && TYPE.equals(parser.getName())) {
				type = BazaarConflictType.fromString(parser.nextText());
			} else if (eventType == XmlPullParser.START_TAG && PATH.equals(parser.getName())) {
				path = parser.nextText();
			} else if (eventType == XmlPullParser.START_TAG && MESSAGE.equals(parser.getName())) {
				message = parser.nextText();
			}
			eventType = parser.next();
		}
		return new BazaarConflict(type, path, message);
	}
}
