package org.vcs.bazaar.client.commandline.parser;

import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.kxml2.io.KXmlParser;
import org.vcs.bazaar.client.BazaarItemInfo;
import org.vcs.bazaar.client.BazaarItemKind;
import org.vcs.bazaar.client.BazaarStatusKind;
import org.vcs.bazaar.client.IBazaarItemInfo;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

public class XMLLsParser extends XMLParser {

	private final static Log LOG = LogFactory.getLog(XMLLsParser.class);

	private final static String LIST = "list";
	private final static String KIND = "kind";
	private final static String PATH = "path";
	private final static String ID   = "id";
	private final static String ITEM = "item";
	private final static String STATUS_KIND = "status_kind";
	private final static String PATTERN = "pattern";


	/**
	 * Parse an XML string into a list of {@link IBazaarItemInfo}s. Parse errors
	 * will be logged, and an empty list will be returned.
	 * @param xml The XML string to parse.
	 * @return The parse result, or an empty list on parse errors.
	 */
	public static List<IBazaarItemInfo> parse(String xml) {
		KXmlParser parser = new KXmlParser();
		try {
			parser.setInput(new StringReader(xml));
			return new XMLLsParser().parse(parser);
		} catch (XmlPullParserException e) {
			LOG.error("XML parsing failed", e);
		}
		return new ArrayList<IBazaarItemInfo>(0);
	}

	private List<IBazaarItemInfo> parse(KXmlParser parser) {
		this.parser = parser;

		List<IBazaarItemInfo> result = new ArrayList<IBazaarItemInfo>();

		try {
			int eventType = parser.getEventType();

			while (eventType != XmlPullParser.END_DOCUMENT) {

				if(eventType == XmlPullParser.START_TAG && !parser.getName().equals(LIST) && parser.getName().equals(ITEM)) {
					IBazaarItemInfo item = parseItem();
					if(item != null) {
						result.add(item);
					}
				}

				eventType = parser.next();
			}
		} catch (XmlPullParserException e) {
			LOG.error("XML parsing failed", e);
		} catch (IOException e) {
			LOG.error("IO trouble while parsing XML", e);
		}
		return result;
	}

	private IBazaarItemInfo parseItem() throws XmlPullParserException, IOException {

		BazaarItemKind kind = null;
		String id = null;
		String path = null;
		BazaarStatusKind type = null;
		String pattern = null;

		int eventType = parser.getEventType();
		do {
			switch(eventType) {
			case XmlPullParser.START_TAG:
				if(KIND.equals(parser.getName())) {
					kind = BazaarItemKind.fromString(parser.nextText());
				}
				if(PATH.equals(parser.getName())) {
					path = parser.nextText();
				}
				if(ID.equals(parser.getName())) {
					id = parser.nextText();
				}
				if(STATUS_KIND.equals(parser.getName())) {
					type = BazaarStatusKind.fromString(parser.nextText());
				}
				if(PATTERN.equals(parser.getName())) {
					pattern = parser.nextText();
				}
				break;
			}
		} while (!(XmlPullParser.END_TAG == (eventType = parser.next()) && ITEM.equals(parser.getName())));

		return new BazaarItemInfo(kind, id, path, type, pattern);

	}

}
