/**
 * LICENSE + COPYRIGHT
 */
package org.vcs.bazaar.client.commandline.parser;

import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.kxml2.io.KXmlParser;
import org.vcs.bazaar.client.BazaarStatusKind;
import org.vcs.bazaar.client.IBazaarLogMessage;
import org.vcs.bazaar.client.IBazaarStatus;
import org.vcs.bazaar.client.commandline.CommandLineStatus;
import org.vcs.bazaar.client.core.BazaarClientException;
import org.vcs.bazaar.client.utils.StringUtil;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

/**
 * I'm a parser for the output generated by th command: status --xml (usign
 * bzr-xmloutput plugin), and with that output I create a
 * Set&lt;IBazaarStatus&gt;
 *
 * @author Guillermo Gonzalez
 */
public class XMLStatusParser extends XMLParser {

	private final static Log LOG = LogFactory.getLog(XMLStatusParser.class);

	private final List<IBazaarStatus> statuses = new ArrayList<IBazaarStatus>(0);
	private List<IBazaarLogMessage> pendingMerges = null;
	private Set<IBazaarStatus> statusSet = null;

	private final static String ADDED = "added";
	private final static String REMOVED = "removed";
	private final static String RENAMED = "renamed";
	private final static String MODIFIED = "modified";
	private final static String KIND_CHANGED = "kind-changed";
	private final static String UNKNOWN = "unknown";
	private final static String FILE = "file";
	private final static String DIR = "directory";
	private final static String OLDKIND = "oldkind";
	private final static String NEWKIND = "newkind";
	private final static String OLDPATH = "newkind";
	private final static String FID = "fid";
	private final static String SUFFIX = "suffix";
	private final static String STATUS = "status";
	private final static String BRANCH_ROOT = "workingtree_root";
	private final static String CONFLICTS = "conflicts";
	private final static String CONFLICT = "conflict";
	private final static String PENDING_MERGES = "pending_merges";

	public void parse(final String xml) throws BazaarClientException {
		parser = new KXmlParser();
		try {
			parser.setInput(new StringReader(xml));
			int eventType = parser.getEventType();
			while (eventType != XmlPullParser.END_DOCUMENT) {
				if (eventType == XmlPullParser.START_TAG && isGroup(parser.getName())) {
					parseGroup();
				} else if (eventType == XmlPullParser.START_TAG && STATUS.equals(parser.getName())) {
					workDir = new File(parser.getAttributeValue(null, BRANCH_ROOT));
				} else if (eventType == XmlPullParser.START_TAG && PENDING_MERGES.equals(parser.getName())) {
					parsePendingMerges();
				}
				eventType = parser.next();
			}
		} catch (XmlPullParserException e) {
			throw BazaarClientException.wrapException(e);
		} catch (IOException e) {
			throw BazaarClientException.wrapException(e);
		}
		statusSet = orderAndCleanup();
	}

	public Set<IBazaarStatus> getStatusSet() {
		return statusSet;
	}

	public List<IBazaarLogMessage> getPendingMerges() {
		if(pendingMerges == null) {
			pendingMerges = Collections.emptyList();
		}
		return pendingMerges;
	}

	private void parsePendingMerges() throws BazaarClientException {
		LOG.debug("Parsing pending merges");
		try {
			pendingMerges = new XMLLogParser().parse(parser);
		} catch (Exception e) {
			LOG.error("unexpected error while parsing pending merges", e);
			pendingMerges = null;
		}
		if(pendingMerges == null) {
			pendingMerges = new ArrayList<IBazaarLogMessage>(0);
		}
	}

	private boolean isGroup(String name) {
		return (name.equals(ADDED) || name.equals(REMOVED) || name.equals(RENAMED)
				|| name.equals(MODIFIED) || name.equals(KIND_CHANGED)
				|| name.equals(UNKNOWN)) || name.equals(CONFLICTS);
	}

	private void parseGroup() throws XmlPullParserException, IOException {
		String group = parser.getName();
		LOG.debug("Parsing status group: " + group);
		int eventType = parser.next();
		while (eventType != XmlPullParser.END_DOCUMENT) {
			if(eventType == XmlPullParser.START_TAG && CONFLICT.equals(parser.getName())) {
				final String path = StringUtil.nullSafeTrim(parser.nextText());
				if((path.endsWith(".BASE") || path.endsWith(".THIS") || path.endsWith(".OTHER"))) {
					eventType = parser.next();
					continue;
				}
				// TODO: add conflict type enum
				final IBazaarStatus status = new CommandLineStatus(getStatusKind(group), getAsFile(path), false, null, null, null, workDir);
				statuses.add(status);
			} else if (eventType == XmlPullParser.START_TAG && (parser.getName().equals(FILE) || parser.getName().equals(DIR))) {
				String path, prevPath, newKind, oldKind;
				path = prevPath = newKind = oldKind = null;
				for (int i = 0; i < parser.getAttributeCount(); i++) {
					if (parser.getAttributeName(i).equals(OLDKIND)) {
						oldKind = StringUtil.nullSafeTrim(parser.getAttributeValue(i));
					} else if (parser.getAttributeName(i).equals(NEWKIND)) {
						newKind = StringUtil.nullSafeTrim(parser.getAttributeValue(i));
					} else if (parser.getAttributeName(i).equals(OLDPATH)) {
						prevPath = StringUtil.nullSafeTrim(parser.getAttributeValue(i));
					} else if (parser.getAttributeName(i).equals(FID)) {
						// do nothing (for the moment)
					} else if (parser.getAttributeName(i).equals(SUFFIX)) {
						// do nothing (for the moment)
					}
				}
				path = StringUtil.nullSafeTrim(parser.nextText());
				if(path.endsWith(".BASE") || path.endsWith(".THIS") || path.endsWith(".OTHER")) {
					eventType = parser.next();
					continue;
				}
				final IBazaarStatus status = new CommandLineStatus(
						getStatusKind(group), getAsFile(path), parser.getName().equals(DIR),
						getAsFile(prevPath), newKind, oldKind, workDir);
				statuses.add(status);
			} else if (eventType == XmlPullParser.END_TAG && group.equals(parser.getName())) {
				return;
			}
			eventType = parser.next();
		}
	}

	private static BazaarStatusKind getStatusKind(final String group) {
		if (group.equals(ADDED)) {
			return BazaarStatusKind.CREATED;
		} else if (group.equals(REMOVED)) {
			return BazaarStatusKind.DELETED;
		} else if (group.equals(RENAMED)) {
			return BazaarStatusKind.RENAMED;
		} else if (group.equals(MODIFIED)) {
			return BazaarStatusKind.MODIFIED;
		} else if (group.equals(KIND_CHANGED)) {
			return BazaarStatusKind.KIND_CHANGED;
		} else if (group.equals(UNKNOWN)) {
			return BazaarStatusKind.UNKNOWN;
		} else if (group.equals(CONFLICTS)) {
			return BazaarStatusKind.HAS_CONFLICTS;
		}
		return null;
	}

	private static File getAsFile(final String relativePathTofile) {
		if(relativePathTofile != null && !"".equals(relativePathTofile)) {
			return new File(relativePathTofile);
		}
		return null;
	}

	private Set<IBazaarStatus> orderAndCleanup() {
		final Map<String, List<IBazaarStatus>> map = new HashMap<String, List<IBazaarStatus>>();
		for (IBazaarStatus status : statuses) {
			List<IBazaarStatus> list = map.get(status.getPath());
			if (list == null) {
				list = new ArrayList<IBazaarStatus>();
				map.put(status.getPath(), list);
			}
			list.add(status);
		}
		return unifyStatuses(map);
	}

	private static Set<IBazaarStatus> unifyStatuses(Map<String, List<IBazaarStatus>> map) {
		final Set<String> keySet = map.keySet();
		final Set<IBazaarStatus> set = new HashSet<IBazaarStatus>(keySet.size());
		for (String key : keySet) {
			CommandLineStatus keeped = null;
			for (IBazaarStatus status : map.get(key)) {
				if (keeped == null) {
					keeped = (CommandLineStatus) status;
				} else {
					keeped.merge(status);
				}
			}
			set.add(keeped);
		}
		return set;
	}

	/**
	 *
	 * @param logParser
	 * @param endTag
	 * @return a List<IBazaarStatus>. Not null.
	 * @throws XmlPullParserException
	 * @throws IOException
	 */
	public List<IBazaarStatus> parseForLog(final KXmlParser logParser, final String endTag) throws XmlPullParserException, IOException {
		statuses.clear();
		parser = logParser;
		int eventType = parser.next();
		while ((!endTag.equals(parser.getName()) && eventType != XmlPullParser.END_TAG )) {
			if (eventType == XmlPullParser.START_TAG && isGroup(parser.getName())) {
				parseGroup();
			}
			eventType = parser.next();
		}
		final Set<IBazaarStatus> mergedSet = orderAndCleanup();
		return Arrays.asList(mergedSet.toArray(new IBazaarStatus[mergedSet.size()]));
	}

}
