// Copyright (c) 2004-2006 by Leif Frenzel - see http://leiffrenzel.de
package org.vcs.bazaar.client.utils;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URLEncoder;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.regex.Pattern;

/**
 * <p>
 * helper with common string operations.
 * </p>
 *
 * @author Leif Frenzel
 * @author Isaac Devine
 * @author Guillermo Gonzalez <guillo.gonzo AT gmail DOT com>
 */
public class StringUtil {

	private static Pattern badchars = Pattern.compile("[^a-zA-Z0-9.,:/\\\\_~-]");

	public static URI getAbsoluteURI(final String text) {
		// Use a File Object to convert the path into a opaque URI see the URI
		// javadoc. The use of the File object adds the current path to the string uri,
		// moving this to URI.create
		// fix Bug #294061. we need to scape all spaces to make it java.net.URI "friendly"
		String escapedText = text.trim().replace(" ", "%20");
		if(BazaarUtilities.isWindows() && escapedText.length() > 1) {
			if(escapedText.substring(0, 2).matches("[a-zA-z]:")) {
				return getAbsoluteURI(URI.create("file:///".concat(escapedText.replace("\\", "/"))));
			}
		}
		return getAbsoluteURI(URI.create(escapedText));
	}

	public static URI getAbsoluteURI(final File file) {
		URI proposed = file.toURI();
		return getAbsoluteURI(proposed);
	}

	private static void urlEncodeBuffer(char[] convertionBuffer, int bufferLength, StringBuffer result) throws UnsupportedEncodingException {
		String toConvert = new String(convertionBuffer, 0, bufferLength);		
		String encoded = URLEncoder.encode(toConvert, "UTF-8");
		result.append(encoded);
	}

	public static String urlEncode(String url) throws UnsupportedEncodingException {

		char[] convertionBuffer = new char[url.length()];
		StringBuffer result = new StringBuffer();
 		int bufferLength = 0;
 		for (char ch : url.toCharArray()) {
 			if ((int)ch > 127) {
 				convertionBuffer[bufferLength++] = ch;
 			} else {
 				if (bufferLength > 0) {
 					urlEncodeBuffer(convertionBuffer, bufferLength, result);
 					bufferLength = 0;
 				}
 				result.append(ch); 			
 			}
 		}
 		if (bufferLength > 0) {
 			urlEncodeBuffer(convertionBuffer, bufferLength, result);
 		}
		return result.toString();
	}

	public static URI getAbsoluteURI(final URI uri) {
		URI proposed = uri;
		String schemeSpecificPart = makeDriveLetterLowerCase(proposed.getRawSchemeSpecificPart());
		if(proposed.getScheme() == null) {
			// the URI don't have shceme (possibly a raw path to the local file system)
			return proposed;
		} else {
			StringBuilder sb = new StringBuilder(":");
			for (int i = 2; i >= 0 ; i--) {
				if(i == 2 && !"file".equals(proposed.getScheme())) {
					continue;
				} else if(schemeSpecificPart.length() <= 2 && i == 2) {
					sb.append("/");
				} else if(schemeSpecificPart.charAt(i) != '/') {
					sb.append("/");
				}
			}
			try {
				schemeSpecificPart = urlEncode(schemeSpecificPart);
			} catch (UnsupportedEncodingException e) {
			}
			proposed = URI.create(proposed.getScheme() + sb.toString() + schemeSpecificPart); //$NON-NLS-1$
			return proposed;
		}
	}

	private static String makeDriveLetterLowerCase(final String str) {
		String result = str;
		// the strange thing is that uris generated with File objects sometimes
		// have uppercase and sometimes lower case drive letters on windows
		// (which doesn't exactly help in comparisons with equals() ...)
		int colonIndex = str.indexOf(':');
		if (colonIndex > 0) {
			final StringBuilder sb = new StringBuilder(str.substring(0, colonIndex - 1));
			sb.append(Character.toLowerCase(str.charAt(colonIndex - 1)));
			sb.append(str.substring(colonIndex));
			result = sb.toString();
		}
		return result;
	}

	public static String[] addToArray(String[] array, String s) {
		String[] newArray = new String[array.length + 1];
		System.arraycopy(array, 0, newArray, 0, array.length);
		newArray[newArray.length - 1] = s;
		return newArray;
	}

	public static String shellQuote(String part) {
		if(part != null && badchars.matcher(part).find()) {
			return "\"" + part + "\"";
		} else {
			return part;
		}
	}

	/**
	 * FIXME: The formatter should parse: Sun 2007-06-24 19:36:42 -0300, this is a
	 * workaround to the problem (see method parseLogDate)
	 */
	private static final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss Z");

	public synchronized static Date parseDate(String date) throws ParseException {
		return DATE_FORMAT.parse(date);
	}

	public synchronized static Date parseLogDate(String date) throws ParseException {
		return DATE_FORMAT.parse(date.substring(4).trim());
	}

	public static String nullSafeTrim(String aString) {
		if(aString == null) {
			return null;
		}
		return aString.trim();
	}

	/* Extremely simple URI escape helper */
	public static String encodeURI(String s) throws UnsupportedEncodingException {
		return urlEncode(s.replaceAll("\\\\", "%5C").replaceAll(" ", "%20"));
	}
}
