package org.vcs.bazaar.client.core;

import static org.junit.Assert.assertEquals;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.URI;
import java.net.URISyntaxException;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

public class BranchLocationTest {

	private File tempDir;
	private String branchDir;
	private File theFile;

	@Before
	public void setUp() {
		tempDir = new File(System.getProperty("java.io.tmpdir"));
		branchDir = "C:/Documents and Settings/dangoore/My Documents/EclipseBZRWorkspace/MyProject";
		theFile = new File(tempDir, branchDir);
		theFile.mkdirs();
	}

	@After
	public void tearDown() {
		theFile.delete();
	}

	@Test
	public void testFilenameEncoding() throws BazaarClientException {
		BranchLocation l1 = new BranchLocation(theFile);
		BranchLocation l2 = new BranchLocation(theFile.getAbsolutePath());
		assertEquals(theFile.getAbsolutePath(), l1.toString());
		assertEquals(theFile.getAbsolutePath(), l2.toString());
	}

	@Test
	public void testLPEncoding() throws BazaarClientException {
		BranchLocation l1 = new BranchLocation("lp:bzr-java-lib");
		assertEquals("lp:bzr-java-lib", l1.toString());
	}

	@Test
	public void testURIEncoding() throws BazaarClientException, URISyntaxException {
		BranchLocation l1 = new BranchLocation(new URI("bzr+ssh://bazaar.launchpad.net/bzr-java-lib/bzr-java-lib/"));
		BranchLocation l2 = new BranchLocation("http://bazaar.launchpad.net/bzr%20java-lib/bzr-java-lib/");
		assertEquals("bzr+ssh://bazaar.launchpad.net/bzr-java-lib/bzr-java-lib/", l1.toString());
		assertEquals("http://bazaar.launchpad.net/bzr%20java-lib/bzr-java-lib/", l2.toString());
		assertEquals("http://bazaar.launchpad.net/bzr%20java-lib/bzr-java-lib/", l2.getUri().toString());
	}

	@Test
	public void testTrim() throws BazaarClientException {
		BranchLocation l1 = new BranchLocation("lp:bzr-java-lib ");
		assertEquals("lp:bzr-java-lib", l1.toString());
	}

	private BranchLocation serializeDeserialize(BranchLocation l) throws Exception {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		ObjectOutputStream oos = new ObjectOutputStream(baos);
		oos.writeObject(l);
		ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
		ObjectInputStream ois = new ObjectInputStream(bais);
		return (BranchLocation) ois.readObject();
	}
	
	@Test
	public void testSerialization() throws Exception {
		BranchLocation l1 = new BranchLocation("lp:bzr-java-lib");
		BranchLocation l2 = new BranchLocation(new URI("bzr+ssh://bazaar.launchpad.net/bzr-java-lib/bzr-java-lib/"));
		BranchLocation l3 = new BranchLocation(theFile);
		
		BranchLocation dl1 = serializeDeserialize(l1);
		BranchLocation dl2 = serializeDeserialize(l2);
		BranchLocation dl3 = serializeDeserialize(l3);
		assertEquals(l1.getScheme(), dl1.getScheme());
		assertEquals(l1.getRaw(), dl1.getRaw());
		assertEquals(l2.getScheme(), dl2.getScheme());
		assertEquals(l2.getUri(), dl2.getUri());
		assertEquals(l3.getScheme(), dl3.getScheme());
		assertEquals(l3.getUri(), dl3.getUri());
	}
	

}
