/**
 * LICENSE + COPYRIGHT
 */
package org.vcs.bazaar.client.commandline.internal;

import static org.vcs.bazaar.client.utils.StringUtil.shellQuote;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;

import org.vcs.bazaar.client.commandline.CmdLineClientTexts;
import org.vcs.bazaar.client.commandline.CommandLineException;
import org.vcs.bazaar.client.commandline.ICommand;
import org.vcs.bazaar.client.commandline.commands.options.Option;
import org.vcs.bazaar.client.core.BazaarClientException;
import org.vcs.bazaar.client.utils.BazaarUtilities;

/**
 * @author Guillermo Gonzalez
 *
 * Abstract base class for command requests. Provides a framework for
 * implementing command execution.
 */
public abstract class Command implements ICommand {

	private CommandRunner runner;
	protected File workDir;
	protected List<Option> options;
	protected boolean checkExitValue = true;

	protected Command() {
		super();
		this.options = new ArrayList<Option>();
	}

	protected Command(File workDir) {
		this();
		this.workDir = workDir;
	}

	/**
	 *
	 * @return A String that represent the bzr command name
	 */
	public abstract String getCommand();

	public String getCommandInfo() throws BazaarClientException {
		return CmdLineClientTexts.bind("Command.executing", new String[] { getCmdLineAsString() });
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.vcs.bazaar.client.core.model.ICommand#getEstimatedWork()
	 */
	public int getEstimatedWork() {
		// defaut estimated work
		return 100;
	}

	/**
	 * <p>
	 * Do the real work of a command
	 * </p>
	 * <p>
	 * this method is called in {@link #run(IProgressMonitor)}
	 * </p>
	 * <p>
	 * Subclases must override this method when needed.<br>
	 * Implementors must ensure that the any return value 'll be recorded
	 * </p>
	 *
	 * @throws BazaarClientException
	 */
	public void execute(CommandRunner runner) throws BazaarClientException {
		try {
			this.runner = runner;
			this.runner.runCommand(this, workDir);
		} catch (IOException e) {
			throw new CommandLineException(e, getCommand());
		} catch (InterruptedException e) {
			throw new CommandLineException(e, getCommand());
		}
	}

	/**
	 * Constucts the Bzr command invocation string corresponding to the
	 * arguments.
	 *
	 * @param arguments
	 *
	 * @return the command invocation string
	 * @throws CommandLineException
	 */
	public List<String> constructCommandInvocationString() throws CommandLineException {

		List<String> arguments = getArguments();
		List<String> commandLine = new ArrayList<String>();
		commandLine.addAll(defaultCmdLine());
		for (Option option : options) {
			if(option == null) {
				// only to show a more detailed error when an option is null
				CommandLineException cle = new CommandLineException("Command option cannot be null", getCommand());
				cle.setStackTrace(CommandLineException.getCurrentStackTrace());
				throw cle;
			}
			String strOption = option.toString();
			if (strOption.length() > 0) {

				commandLine.add(strOption);
				//commandLine[options.indexOf(option) + defaultCmdLineSize()] = strOption;
				//lastIdx += 1;
			}
		}

		commandLine.addAll(arguments);

		return commandLine;
	}

	public void setOption(Option option) {
		if (!this.options.contains(option)) {
			this.options.add(option);
		}
	}

	public String getCommandError() {
		return CmdLineClientTexts.bind("Command.error", new String[] { getCommand() });
	}

	public String getStandardError() {
		return runner.getStandardError();
	}

	public String getStandardOutput() {
		return runner.getStandardOutput();
	}

	public String getStandardOutput(String charsetName) throws UnsupportedEncodingException {
		return runner.getStandardOutput(charsetName);
	}

	public String[] getStandardOutputSplit() {
		return runner.getStandardOutputSplit();
	}

	// ///////////////////////////////////////
	// Command line Helpers
	/**
	 * @return cmdLine String[] with empty option of the specified size
	 */
	protected List<String> defaultCmdLine() {
		List<String> result = new ArrayList<String>();
		result.add(getCommand());
		return result;
	}

	/*
	protected int defaultCmdLineSize() {
		String[] defaultCmdLine = new String[] { getExecutable(), getCommand() };
		return defaultCmdLine.length;
	}*/

	protected abstract List<String> getArguments() throws CommandLineException;

	/*
	 * Convenience methods
	 */
	protected List<String> getArguments(String arg) {
		List<String> args = new ArrayList<String>();
		if(arg != null) {
			args.add(arg);
		}
		return args;
	}

	protected List<String> getArguments(String arg1, String arg2) {
		List<String> args = getArguments(arg1);
		args.add(arg2);
		return args;
	}

	protected List<String> getEmptyArguments() {
		return new ArrayList<String>();
	}

	/**
	 * @return a the cmd line that is going to be executed
	 * @throws CommandLineException
	 */
	protected String getCmdLineAsString() throws CommandLineException {
		final StringBuilder cmdLine = new StringBuilder();
		for (String part : constructCommandInvocationString()) {
			cmdLine.append(shellQuote(part));
			cmdLine.append(" ");
		}
		return cmdLine.toString();
	}

	protected File getRelativeToWorkDir(File file) {
		return BazaarUtilities.getRelativeTo(this.workDir, file);
	}

	public void setEnviroment(ProcessBuilder pb) {
	}

	public boolean isCheckExitValue() {
		return checkExitValue;
	}
}
