package org.vcs.bazaar.client.core;

import org.vcs.bazaar.client.commandline.CommandLineException;

/**
 * A checked expection representing a failure in Bazaar.
 * <p>
 * </p>
 */
public class BazaarClientException extends Exception {

	private static final long serialVersionUID = -3048005644862762989L;

	private String error;
	private String info;

	/**
	 * @param aCheckedException
	 * @return
	 */
	public static RuntimeException makeUnChecked(final Exception exception) {
		return new BazaarUncheckedException(exception);
	}

	public static class BazaarUncheckedException extends RuntimeException {
		private static final long serialVersionUID = -4469538621094655850L;

		public BazaarUncheckedException(Exception exception) {
			super(exception.getMessage(), exception.getCause());
			this.setStackTrace(exception.getStackTrace());
		}

		public BazaarUncheckedException(String message) {
			super(message);
		}
	}

	/**
	 * @param e
	 */
	public static BazaarClientException wrapException(Exception exception) {
		return new BazaarClientException(exception.getMessage(), exception);
	}

	/**
	 * @param e
	 */
	public static BazaarClientException wrapException(CommandLineException exception) {
		return new BazaarClientException(exception);
	}

	public BazaarClientException(Exception exception) {
		super(exception.getMessage(), exception.getCause());
	}

	public BazaarClientException(String message) {
		super(message);
	}

	public BazaarClientException(String message, Throwable t) {
		super(message, t);
		this.setStackTrace(t.getStackTrace());
	}

	public BazaarClientException(CommandLineException e) {
		this(e.getMessage(), e);
		this.error = e.getMessageError();
		this.info = e.getMessageInfo();
	}

	public String getError() {
		return error;
	}

	public String getInfo() {
		return info;
	}

	public boolean isCommandLine() {
		return getCause() instanceof CommandLineException;
	}

	/**
	 * Returns the command that caused this exception only if this contains a CommandLineException.
	 * @return
	 */
	public String getCommand() {
		if(getCause() instanceof CommandLineException) {
			return ((CommandLineException)getCause()).getCommand();
		} else {
			return null;
		}
	}

	/**
	 * Create a stacktrace from the current thread, and remove the calls used to
	 * generate it
	 *
	 * @return StackTraceElement[]
	 */
	public static StackTraceElement[] getCurrentStackTrace() {
		// a custom stackTrace (all calls except
		// Thread.currentThread().getStackTrace() ;) )
		StackTraceElement[] st = Thread.currentThread().getStackTrace();
		System.arraycopy(st, 3, st, 0, st.length - 3);
		return st;
	}

}
