/*******************************************************************************
 * Copyright (c) 2012 Idereal Limited, Piotr Piastucki
 *
 * Contributors:
 *    Alexander Taler
 *    Guillermo Gonzalez
 *    Piotr Piastucki
 *******************************************************************************/
package org.vcs.bazaar.client.core;

import java.io.File;
import java.io.Serializable;
import java.net.URI;

import org.vcs.bazaar.client.utils.StringUtil;

public class BranchLocation implements Comparable<BranchLocation>, Serializable {

	private static final long serialVersionUID = 6174757231086380623L;
	
	Scheme scheme;  // The Locator scheme
    URI uri = null;    // For schemes that have URIs, this is used.
    String raw = null; // Otherwise the raw creation string is used.

    private static class Scheme implements Serializable {
		private static final long serialVersionUID = 1L;
		Scheme(String i, boolean w) {
    		id = i;
    		writeable = w;
    	}
    	String id;
    	boolean writeable;
    	
    }
    
    private static final Scheme[] SCHEMES = {
    	new Scheme("file", true),
    	new Scheme("bzr", true),
    	new Scheme("bzr+ssh", true),
    	new Scheme("lp", true),
    	new Scheme("aftp", true),
    	new Scheme("sftp", true),
    	new Scheme("ftp", true),
    	new Scheme("rsync", true),
    	new Scheme("http", false),
    	new Scheme("https", false),
    	new Scheme("bzr+http", false),
    	new Scheme("bzr+https", false),
    	new Scheme("https+urllib", false),
    };
    
    private static Scheme lookupScheme(String schemeId) throws BazaarClientException {
        for (Scheme s : SCHEMES) {
        	if (s.id.equals(schemeId)) {
        		return s;
        	}
        }
        throw new BazaarClientException("Unsupported Scheme");
    }

    /**
     * Create a new Locator object for the given string.
     *
     * The string can be a path to the local filesystem, or a URI string which
     * Bazaar would accept.  The windows drive letter prefix [A-Z]: will be
     * treated as a local file instead of a URL.
     * 
     * @param raw
     * @throws BazaarException if the string was not a valid locator
     */
    public BranchLocation(String raw) throws BazaarClientException {
        if (raw == null || raw.trim().isEmpty()) {
        	throw new BazaarClientException("Branch Not Specified");
        }
        
        raw = raw.trim();
    	this.raw = raw;
        
        if (raw.startsWith("lp:")) {
            if (raw.length() == 3) {
            	throw new BazaarClientException("Branch Not Specified");
            }
        	scheme = lookupScheme("lp");
        	return;
        }
        if (!raw.matches("^[a-zA-Z]:.*")) {
            // Try it as a URI
        	try {
                uri = new URI(StringUtil.encodeURI(raw));
                if (uri.getScheme() != null) {
                	scheme = lookupScheme(uri.getScheme());
                	return;
                }
            } catch (Exception e) {
                // Fallback to file
        	}
        }
        initAsFile(new File(raw));
    }
    
    /**
     * Create a new Locator object for the given URI.
     *
     * If the URI has no scheme, or looks like a Windows drive letter scheme
     * [A-Z]: then it will be treated as a local file reference.
     * 
     * TODO: Check for the existence of file: URIs on local filesystem
     *
     * @param uri
     * @throws BazaarException if the URI is not appropriate
     */
    public BranchLocation(URI uri) throws BazaarClientException {
        this.uri = uri;

        if (uri == null) {
        	throw new BazaarClientException("Branch Not Specified");
        }

        String schemeId = uri.getScheme();
        // If the URI is relative or looks like a Windows file, fallback to file:
        if ((schemeId == null) || (schemeId.matches("^[a-zA-Z]$"))) {
            initAsFile(new File(uri.getPath()));
        } else {
            scheme = lookupScheme(schemeId);
        }
    }

	public BranchLocation(File file) throws BazaarClientException {
        if (file == null) {
        	throw new BazaarClientException("Branch Not Specified");
        }
        initAsFile(file);
	}
    
    /**
     * Initialise a new object as a local file reference
     * @return 
     */
    private void initAsFile(File file) throws BazaarClientException {
        // Try and find the file on local filesystem
        File parent = file.getParentFile();
        if (!file.exists() && (parent == null || !parent.exists())) {
            throw new BazaarClientException("File not found");
        }
        raw = file.getAbsolutePath();
        scheme = lookupScheme("file");
        uri = file.toURI();
    }

    public boolean isWriteable() {
    	return scheme.writeable;
    }

    @Override
	public String toString() {
    	if (raw != null) {
            return raw;
        } 
		return uri.toString();
    }
    
	public int compareTo(BranchLocation other) {
		String location = toString();
		if(location == null) {
			return -1;
		} else {
			return location.compareTo(other.toString());
		}
	}

	@Override
	public int hashCode() {
		String location = toString();
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((location == null) ? 0 : location.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		BranchLocation other = (BranchLocation) obj;
		String location = toString();
		if (location == null) {
			if (other.toString() != null)
				return false;
		} else if (!location.equals(other.toString()))
			return false;
		return true;
	}

	public String getScheme() {
		return scheme.id;
	}

	public URI getUri() {
		return uri;
	}

	public String getRaw() {
		return raw;
	}

}
