/**
 * LICENCSE + COPYRIGHT
 */
package org.vcs.bazaar.client.utils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.StringTokenizer;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.vcs.bazaar.client.BazaarClientFactory;
import org.vcs.bazaar.client.BazaarClientPreferences;
import org.vcs.bazaar.client.BazaarPreference;
import org.vcs.bazaar.client.IPlugin;
import org.vcs.bazaar.client.commandline.commands.Plugins;
import org.vcs.bazaar.client.commandline.internal.ShellCommandRunner;
import org.vcs.bazaar.client.commandline.parser.XMLPluginParser;
import org.vcs.bazaar.client.core.BazaarClientException;

/**
 * @author Guillermo Gonzalez <guillo.gonzo@gmail.com>
 *
 */
public final class BazaarUtilities {

	private static final Log LOG = LogFactory.getLog(BazaarUtilities.class);

    /**
     * The Unix separator character.
     */
    private static final char UNIX_SEPARATOR = '/';

    /**
     * The Windows separator character.
     */
    private static final char WINDOWS_SEPARATOR = '\\';

    protected static final boolean isWindows() {
    	return WINDOWS_SEPARATOR == File.separatorChar;
    }

	/**
	 * @param file
	 * @return
	 */
	public static File getRootBranch(File file) {
		while (file.isFile() || !file.exists()) {
			file = file.getParentFile();
		}

		// find .bzr in this directory
		File[] files = file.listFiles(new FileFilter() {
			public boolean accept(File pathname) {
				return pathname.isDirectory() && pathname.getName().equals(".bzr");
			}
		});

		if (files.length == 0) { // not found? try parent (if there is any)!
			File parent = file.getParentFile();
			return (parent != null) ? getRootBranch(parent) : null;

		} else { // found!
			return file;
		}

	}

	public final static File getRelativeTo(File base, File file) {
		String normalizedBase = unixFilePath(base);
		String normalizedFile = unixFilePath(file);
		String[] workDirElements = normalizedBase.split(""+UNIX_SEPARATOR);
		String[] fileElements = normalizedFile.split(""+UNIX_SEPARATOR);
		File relativeFile = file;
		int elementIdx = 0;
		while(elementIdx < fileElements.length) {
			if(elementIdx < workDirElements.length && workDirElements[elementIdx].equals(fileElements[elementIdx])) {
				elementIdx++;
			} else {
				final StringBuilder sb = new StringBuilder();
				for (int i = elementIdx; i < fileElements.length; i++) {
					sb.append(fileElements[i]).append(File.separator);
				}
				relativeFile = new File(sb.toString());
				return relativeFile;
			}
		}
		return file;
	}

    public final static String unixFilePath(File file) {
    	String path = "";
    	if(file != null) {
    		path = file.getPath();
    	}
        if (path == null || path.indexOf(WINDOWS_SEPARATOR) == -1) {
            return path;
        }
        return path.replace(WINDOWS_SEPARATOR, UNIX_SEPARATOR);
    }

    /**
     * Parses bzr.bat and return a list of strings that represents:<br>
     * [Drive:\Path\to\python.exe, Drive:\Path\to\bzr]
     *
     * @param executable
     * @return List<String>
     * @throws IOException
     */
    public final static List<String> getWindowsExecutable(final String executable) throws IOException {
    	File batfile = new File(executable);
		FileReader fr = new FileReader(batfile);
		BufferedReader br = new BufferedReader(fr);
		String exe = br.readLine();
		br.close();
		exe = exe.replaceAll("[\\*%@]", "").trim();
		String drive = exe.substring(0, 2)+"\\\\";
		String[] allparts = exe.split("(?=" + drive + ")");
		List<String> output = new ArrayList<String>();
		for (String allpart : allparts) {
			if(allpart.trim().length() == 0) {
				continue;
			} else {
				output.add(allpart.trim());
			}
		}
		return output;
    }

	/**
	 * Check that the file exists.<br>
	 * If the file is not an absolute path, search it in the PATH env var.
	 * @throws BazaarClientException
	 */
	public static void checkExecutable() throws BazaarClientException {
		String exe = BazaarClientPreferences.getInstance().getString(BazaarPreference.EXECUTABLE);
		if(exe == null || "".equals(exe.trim())) {
			throw new BazaarClientException("Invalid bzr executable: '" + exe + "'");
		}
        if(exe.indexOf(File.separatorChar)>=0) {
            // this is full path
            File f = new File(exe);
            if(f.exists()) {
            	return;
            }

            File fexe = new File(exe+".exe");
            if(fexe.exists()) {
            	return;
            }
            throw new BazaarClientException("There's no such file: "+exe);
        } else {
        	try {
        		if(findExecutable().exists()) {
        			return;
        		}
			} catch (FileNotFoundException e) {
				throw BazaarClientException.wrapException(e);
			}
        }
	}

	public static File findExecutable() throws FileNotFoundException {
		String path = System.getenv("PATH");
		return findExecutable(path);
	}

	protected static File findExecutable(String path) throws FileNotFoundException {
		String[] executableNames = null;
		boolean isWindows = isWindows();
		if(isWindows){
			executableNames = new String[]{"bzr.exe", "bzr.bat"};
			return findExecutableWin(executableNames, path);
		} else {
			executableNames = new String[]{"bzr"};
			String[] parts = path.split(File.pathSeparator);
			for (String part : parts) {
				for (String executableName : executableNames) {
					File executable = new File(new File(part), executableName);
					if(executable.canRead()) {
						return executable;
					}
				}
			}
		}
		throw new FileNotFoundException("bzr executable not found");
	}

	protected static File findExecutableWin(String[] exeNames, String path) throws FileNotFoundException {
        // look in PATH
        path = path + File.pathSeparator + System.getenv("PATHEXT");
        String tokenizedPath = "";
        String delimiter = null;
        if(path!=null) {
        	StringTokenizer tokenizer = new StringTokenizer(path.replace("\\", "\\\\"), ""+File.pathSeparator);
        	while(tokenizer.hasMoreElements()) {
        		final String _dir = tokenizer.nextToken();
                if (delimiter == null) {
                  delimiter = ", ";
                }
                else {
                  tokenizedPath += delimiter;
                }

                tokenizedPath += _dir.replace('\\', '/');

                File dir = new File(_dir);
                for (String exe : exeNames) {
                	File f = new File(dir,exe);
                    if(f.exists()) {
                        return f;
                    }
				}
            }

            tokenizedPath += ".";
        }
        // didn't find it
        StringBuilder exes = new StringBuilder();
        for (String exe : exeNames) {
			exes.append(exe).append(",");
		}
        throw new FileNotFoundException("Could not find any of: " + exes.toString() + " in PATH");
	}

	/**
	 *
	 * @param clazz the class to use as the base to getResourceAsStream the tests.properties file
	 * @param propertiesFile path/URI of a properties file to load the preferences from.
	 */
	public static void configureBazaarClient(Class<? extends Object> clazz, String propertiesFile) {
		LOG.debug("Getting environment settings");
		final BazaarClientPreferences prefs = BazaarClientPreferences.getInstance();
		try {
			InputStream is = clazz.getResourceAsStream(propertiesFile);
			Properties fileProps = new Properties();
			if(is != null) {
				fileProps.load(is);
				prefs.setFrom(fileProps);
			}
		} catch (Exception e) {
			LOG.error("Error loading properties", e);
			throw BazaarClientException.makeUnChecked(e);
		}
		if(prefs.getString(BazaarPreference.EXECUTABLE) == null) {
			prefs.set(BazaarPreference.EXECUTABLE, "bzr");
		}
		LOG.debug("Initializing client factories");
		try {
			BazaarClientFactory.setupBestAvailableBackend(true);
		} catch (BazaarClientException e) {
			LOG.error("Error initializing client factories", e);
			throw BazaarClientException.makeUnChecked(e);
		}
	}

	public static void configureBazaarClient() {
		configureBazaarClient(BazaarUtilities.class, "/tests.properties");
	}


	/**
	 * Execute the plugins command and lookup for the xmloutput plugins
	 * @return
	 * @throws BazaarClientException
	 */
	public static String[] getXmlOutputVersion() throws BazaarClientException {
	    Plugins cmd = new Plugins();
        try {
            cmd.execute(new ShellCommandRunner());
            for (IPlugin plugin : new XMLPluginParser().parse(cmd.getStandardOutput())) {
                if(plugin.getName() != null && plugin.getName().contains("xmloutput")) {
                    return plugin.getVersion();
                }
            }
        } catch (BazaarClientException e) {
            if(e.getMessage() != null && "xmlplugins".equals(e.getCommand()) && e.getMessage().contains("unknown command")) {
                throw new BazaarClientException("xmloutput plugin not found");
            } else {
                throw BazaarClientException.wrapException(e);
            }
        }
        // if we didn't found the plugin...just throw an exception
        throw new BazaarClientException("xmloutput plugin not found");
    }

}
