/**
 * LICENCSE + COPYRIGHT
 */
package org.vcs.bazaar.client;

import java.io.File;
import java.io.InputStream;
import java.net.URI;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.vcs.bazaar.client.commandline.commands.options.Option;
import org.vcs.bazaar.client.commandline.syntax.ICleanTreeOptions;
import org.vcs.bazaar.client.commandline.syntax.IConflictsOptions;
import org.vcs.bazaar.client.commandline.syntax.IMergeOptions;
import org.vcs.bazaar.client.commandline.syntax.IMissingOptions;
import org.vcs.bazaar.client.commandline.syntax.IRebaseAbortOptions;
import org.vcs.bazaar.client.commandline.syntax.IRebaseContinueOptions;
import org.vcs.bazaar.client.commandline.syntax.IRebaseOptions;
import org.vcs.bazaar.client.commandline.syntax.IRebaseToDoOptions;
import org.vcs.bazaar.client.commandline.syntax.IRemoveOptions;
import org.vcs.bazaar.client.commandline.syntax.IShelveListOptions;
import org.vcs.bazaar.client.commandline.syntax.IShelveOptions;
import org.vcs.bazaar.client.commandline.syntax.ITagOptions;
import org.vcs.bazaar.client.commandline.syntax.ITagsOptions;
import org.vcs.bazaar.client.commandline.syntax.IUnShelveOptions;
import org.vcs.bazaar.client.core.BazaarClientException;
import org.vcs.bazaar.client.core.BranchLocation;

/**
 * Bzr operations can be performed by calling methods in this interface.
 * @author Guillermo Gonzalez
 */
public interface IBazaarClient {

	/**
	 * Add specified files or directories.
	 *
	 * @param files
	 * @param options
	 * @throws BazaarClientException
	 */
	public void add(File[] files, Option... options) throws BazaarClientException;

	/**
	 * Get a IBazaarAnnoation which contains the origin of each line in a file
	 *
	 * @param file
	 * @param options
	 * @return IBazaarAnnotation
	 * @throws BazaarClientException
	 */
	public IBazaarAnnotation annotate(File file, Option... options) throws BazaarClientException;

	/**
	 * Create a new copy of a branch.
	 *
	 * @param fromLocation
	 * @param toLocation
	 * @param revision (if null the last revision is branched)
	 * @param options
	 * @throws BazaarClientException
	 */
	public void branch(BranchLocation fromLocation, File toLocation, IBazaarRevisionSpec revision, Option... options) throws BazaarClientException;

	/**
	 * Create a new copy of a branch.
	 *
	 * @param fromLocation
	 * @param toLocation
	 * @param revision (if null the last revision is branched)
	 * @param listener progress listener
	 * @param options
	 * @throws BazaarClientException
	 */
	public void branch(BranchLocation fromLocation, File toLocation, IBazaarRevisionSpec revision, IBazaarProgressListener listener, Option... options) throws BazaarClientException;

	/**
	 * Convert the current branch into a checkout of the supplied {@link BranchLocation}.
	 * @param location
	 * @param options
	 * @throws BazaarClientException
	 * void
	 */
	public void bind(BranchLocation location, Option...options) throws BazaarClientException;

	/**
	 * Get a Reader of the file content from the specified revision.<br>
	 * If revision is null the last revision is used
	 *
	 * @param file A file to read.
	 * @param revision The revision of the file to read, or null to read from
	 * the file's latest revision.
	 * @return a Reader from which the contents of the requested file at the
	 * given revision can be read.
	 * @throws BazaarClientException
	 */
	public InputStream cat(File file, IBazaarRevisionSpec revision, Option... options) throws BazaarClientException;

	/**
	 * @see #cat(File, BazaarRevision, Option...)
	 *
	 * @param file
	 * @param revision
	 * @param charsetName
	 * @param options
	 * @return
	 * @throws BazaarClientException
	 */
	public InputStream cat(File file, IBazaarRevisionSpec revision, String charsetName, Option... options) throws BazaarClientException;

	/**
	 * @see #cat(File, BazaarRevision, Option...)
	 *
	 * @param location Location to read.
	 * @param revision
	 * @param charsetName
	 * @param options
	 * @return
	 * @throws BazaarClientException
	 */
	public InputStream cat(BranchLocation location, IBazaarRevisionSpec revision, String charsetName, Option... options) throws BazaarClientException;
	
	/**
	 * @see #cat(File, BazaarRevision, Option...)
	 *
	 * @param location Location to read.
	 * @param revision
	 * @param charsetName
	 * @param options
	 * @return
	 * @throws BazaarClientException
	 */
	public InputStream cat(URI location, IBazaarRevisionSpec revision, String charsetName, Option... options) throws BazaarClientException;

	/**
	 * Create a new checkout of an existing branch.
	 *
	 * @param fromLocation
	 * @param toLocation
	 * @param options
	 * @throws BazaarClientException
	 */
	public void checkout(BranchLocation fromLocation, File toLocation, Option... options) throws BazaarClientException;

	/**
	 * Create a new checkout of an existing branch.
	 *
	 * @param fromLocation
	 * @param toLocation
	 * @param listener
	 * @param options
	 * @throws BazaarClientException
	 */
	public void checkout(BranchLocation fromLocation, File toLocation, IBazaarProgressListener listener, Option... options) throws BazaarClientException;

	/**
	 * Commit the changes in the specified files
	 *
	 * @param files
	 * @param message
	 * @throws BazaarClientException
	 */
	public void commit(File[] files, String message, Option... options) throws BazaarClientException;

	/**
	 * Show differences in the working tree or between revisions.
	 *
	 * @param file
	 * @throws BazaarClientException
	 */
	public String diff(File[] files, IBazaarRevisionSpec range, Option... options) throws BazaarClientException;

	/**
	 * Find a base revision for merging two branches.
	 * @param branch
	 * @param other
	 * @return
	 * @throws BazaarClientException
	 */
	public BazaarRevision findMergeBase(BranchLocation branch, BranchLocation other) throws BazaarClientException;

	/**
	 * Show information about a working tree, branch or repository.
	 *
	 * @param location
	 * @param options
	 * @return IBazaarInfo
	 * @throws BazaarClientException
	 */
	public IBazaarInfo info(BranchLocation location, Option... options) throws BazaarClientException;

	/**
	 * @see #info(BranchLocation, Option...)
	 */
	public IBazaarInfo info(File location, Option... options) throws BazaarClientException;

	/**
	 * Make a directory into a versioned branch.
	 *
	 * @param location
	 * @param options
	 * @throws BazaarClientException
	 */
	public void init(File location, Option... options) throws BazaarClientException;

	/**
	 * Add the specified file or pattern to .bzrignore
	 *
	 * @param file - a file in the branch (or the branch root)
	 * @param pattern
	 * @throws BazaarClientException
	 */
	public void ignore(File file, String pattern) throws BazaarClientException;

	/**
	 * <p>
	 * Return the ignore patterns and the files affected by them
	 * </p>
	 *
	 * @param file -
	 *            the branch root
	 * @return
	 * @throws BazaarClientException
	 */
	public Map<String, String> ignored(File file) throws BazaarClientException;

	/**
	 * <p>
	 * Get the log of a branch, file, or directory.
	 * </p>
	 *
	 * @param location
	 * @return List<IBazaarLogMessage>
	 * @throws BazaarClientException
	 */
	public List<IBazaarLogMessage> log(File location, Option... options) throws BazaarClientException;

	/**
	 * @see #log(File)
	 */
	public List<IBazaarLogMessage> log(URI location, Option... options) throws BazaarClientException;

	/**
	 * @see #log(File)
	 */
	public List<IBazaarLogMessage> log(BranchLocation location, Option... options) throws BazaarClientException;


	/**
	 * This version provides less options, but it allows clients to
	 * use a hanlder to get "notified" on each log retrieval.
	 *
	 * @see #log(BranchLocation, Option...)
	 *
	 * @param location
	 * @param logHandler
	 * @param limit
	 * @throws BazaarClientException
	 */
	public void log(BranchLocation location, IBzrLogMessageHandler logHandler, Option...options) throws BazaarClientException;

	/**
	 * An async version of {@link #log(BranchLocation, IBzrLogMessageHandler, int)}
	 * The execution is done in a worker thread.
	 *
	 * @param location
	 * @param logHandler
	 * @param limit
	 * @throws BazaarClientException
	 */
	public void logAsync(BranchLocation location, IBzrLogMessageHandler logHandler, Option...options) throws BazaarClientException;

	/**
	 * List files in a tree.
	 * @param workDir
	 * @param options
	 * @return
	 * @throws BazaarClientException
	 * String[]
	 */
	public IBazaarItemInfo[] ls(File workDir, IBazaarRevisionSpec revision, Option...options) throws BazaarClientException;

	/**
	 * List files in a branch.
	 * @param location
	 * @param options
	 * @return
	 * @throws BazaarClientException
	 * String[]
	 */
	public IBazaarItemInfo[] ls(BranchLocation location, IBazaarRevisionSpec revision, Option...options) throws BazaarClientException;

	/**
	 * List files at the specified URL.
	 * @param location
	 * @param options
	 * @return
	 * @throws BazaarClientException
	 * String[]
	 */
	public IBazaarItemInfo[] ls(URI location, IBazaarRevisionSpec revision, Option...options) throws BazaarClientException;

	/**
	 * Perform a three-way merge.
	 *
	 * @param otherBranch
	 * @param options see {@link IMergeOptions}
	 * @throws BazaarClientException
	 */
	public void merge(BranchLocation otherBranch, Option...options) throws BazaarClientException;

	/**
	 * <p>
	 * Move or rename a file.
	 * </p>
	 * <p>
	 * If the last argument is a versioned directory, all the other names are
	 * moved into it. Otherwise, there must be exactly two arguments and the
	 * file is changed to a new name.
	 * </p>
	 *
	 * @param orig
	 * @param dest
	 * @throws BazaarClientException
	 */
	public void move(File[] orig, File dest, Option... options) throws BazaarClientException;

	/**
	 * a convenience method for renames or moves for one file see
	 * {@link #move(File[], File)}
	 *
	 * @param orig
	 * @param dest
	 * @throws BazaarClientException
	 */
	public void move(File orig, File dest, Option... options) throws BazaarClientException;

	/**
	 * Gets the unmerged/unpulled revisions between two branches.
	 * @param workdir the local branch
	 * @param otherBranch the other branch (can be local or remote)
	 * @param options that belongs to {@link IMissingOptions}
	 * @return {@link Map} with only two keys MINE and OTHER, and the values are a list of logs
	 * @throws BazaarClientException
	 */
	public Map<String, List<IBazaarLogMessage>> missing(File workdir, URI otherBranch, Option... options) throws BazaarClientException;

	/**
	 * @see #missing(File, URI, Option...)
	 *
	 * @param workdir
	 * @param otherBranch
	 * @param options
	 * @return
	 * @throws BazaarClientException
	 */
	public Map<String, List<IBazaarLogMessage>> missing(File workdir, BranchLocation otherBranch, Option... options) throws BazaarClientException;

	/**
	 * <p>
	 * If newNick is null return the branch nickname, otherwise set the branch
	 * nickname to the given newNick
	 * </p>
	 *
	 * @param newNick
	 * @return String Branch nickname
	 * @throws BazaarClientException
	 */
	public String nick(String newNick) throws BazaarClientException;

	/**
	 * List the installed plugins.
	 * @return Set<IPlugin>
	 * @throws BazaarClientException
	 */
	public Set<IPlugin> plugins() throws BazaarClientException;

	/**
	 * <p>
	 * Turn this branch into a mirror of another branch.
	 * </p>
	 * <p>
	 * This command only works on branches that have not diverged. Branches are
	 * considered diverged if the destination branch's most recent commit is one
	 * that has not been merged (directly or indirectly) into the parent.
	 * </p>
	 * <p>
	 * If branches have diverged, you can use {@link #merge(URI)} to integrate
	 * the changes from one into the other. Once one branch has merged, the
	 * other should be able to pull it again.
	 * </p>
	 * <p>
	 * If you want to forget your local changes and just update your branch to
	 * match the remote one, use overwrite == true.
	 * </p>
	 *
	 * @throws BazaarClientException
	 */
	public String pull(BranchLocation location, Option... options) throws BazaarClientException;

	/**
	 * see {@link #pull(BranchLocation, Option...)}
	 * @param location as {@link java.net.URI}
	 * @param options
	 * @throws BazaarClientException
	 */
	public String pull(URI location, Option... options) throws BazaarClientException;

	/**
	 * Update a mirror of the branch setted in this.workDir.
	 *
	 * @param location
	 * @param options
	 * @throws BazaarClientException
	 */
	public void push(URI location, Option... options) throws BazaarClientException;

	/**
	 * @see #push(URI, Option...)
	 *
	 * @param location
	 * @param options
	 * @throws BazaarClientException
	 */
	public void push(BranchLocation location, Option... options) throws BazaarClientException;
	/**
	 * <p>
	 * Make the given files unversioned.
	 * </p>
	 * <p>
	 * This makes bzr stop tracking changes to a versioned file. It does not
	 * delete the working copy
	 * </p>
	 * <p>
	 * If onlyAdded is true, only 'added' files will be removed. If you specify
	 * both (files != null && onlyAdded == true), then new files in the
	 * specified directories will be removed. If the directories are also new,
	 * they will also be removed.
	 * </p>
	 *
	 * @param files
	 * @param options
	 *            look at {@link IRemoveOptions} for posible values
	 * @throws BazaarClientException
	 */
	public void remove(File[] files, Option... options) throws BazaarClientException;

	/**
	 * Mark a conflict as resolved.
	 * @param files
	 * @param options
	 * @throws BazaarClientException
	 */
	public void resolve(List<File> files, Option...options) throws BazaarClientException;

	/**
	 *
	 * @param files
	 * @throws BazaarClientException
	 */
	public void revert(File[] files, Option... options) throws BazaarClientException;

	/**
	 * <p>
	 * Returns the current revision for the branch under #location, if location
	 * is null the revno for the branch under 'workDir'. (The file could be the
	 * branch root)
	 * </p>
	 *
	 * @param location
	 * @return BazaarRevision
	 * @throws BazaarClientException
	 */
	public BazaarRevision revno(File location) throws BazaarClientException;

	/**
	 * @see #revno(File)
	 */
	public BazaarRevision revno(BranchLocation location) throws BazaarClientException;

	/**
	 * Create a merge-directive for submiting changes.
	 *
	 * @param submitBranch
	 * @param options
	 * @throws BazaarClientException
	 */
	public void send(BranchLocation submitBranch, Option...options) throws BazaarClientException;

	/**
	 * Returns a BazaarRevision containing the revid: for the nominated
	 * location and revision.
	 * If revision is null, the current tip revision is returned.
	 * @param location
	 * @param revision
	 * @return
	 * @throws BazaarClientException
	 */
	public BazaarRevision revisionInfo(File location, IBazaarRevisionSpec revision) throws BazaarClientException;

	/**
	 * Return the status IBazaarStatus for each file. (is recursive in case a
	 * File is a directory)
	 *
	 * @param files
	 * @return BazaarStatus[]
	 * @throws BazaarClientException
	 */
	public BazaarTreeStatus status(File[] files, Option... options) throws BazaarClientException;

	/**
	 * Set the branch of a checkout and update.
	 *
	 * @param location
	 * @param options
	 * @throws BazaarClientException
	 */
	public void switchBranch(BranchLocation location, Option...options) throws BazaarClientException;

	/**
	 * Convert the current checkout into a regular branch.
	 * @param options
	 * @throws BazaarClientException
	 */
	public void unBind(Option...options) throws BazaarClientException;

	/**
	 * Remove the last committed revision
	 *
	 * @param location
	 *            the branch root
	 * @throws BazaarClientException
	 */
	public void unCommit(File location, Option... options) throws BazaarClientException;

	/**
	 * List unknown files.
	 *
	 * @return String[] which contains the paths or null if the {@link File} is
	 *         not under version control
	 * @throws BazaarClientException
	 */
	public String[] unknowns(File workDir) throws BazaarClientException;

	/**
	 * Update a tree to have the latest code committed to its branch.
	 * @param file - A directory
	 * @param options
	 * @return a message from stdout as String
	 * @throws BazaarClientException
	 */
	public String update(File file, Option...options) throws BazaarClientException;

	/**
	 * Show version information about this tree.
	 * @param location - a branch location
	 * @param options
	 * @return a {@link BazaarVersionInfo}
	 * @throws BazaarClientException
	 */
	public BazaarVersionInfo versionInfo(BranchLocation location, Option...options) throws BazaarClientException;

	/**
	 * Add a notification listener
	 *
	 * @param listener
	 */
	public void addNotifyListener(IBazaarNotifyListener listener);

	/**
	 * Remove a notification listener
	 *
	 * @param listener
	 */
	public void removeNotifyListener(IBazaarNotifyListener listener);

	/**
	 * @return the notification handler
	 */
	public BazaarNotificationHandler getNotificationHandler();

	/**
	 * Add a callback for prompting for username (when neccesary) and password
	 *
	 * @param callback
	 */
	public void addPasswordCallback(IBazaarPromptUserPassword callback);

	/**
	 * Set the working directory (the root of the local branch)
	 *
	 * @param workDir
	 */
	public void setWorkDir(File workDir);

	/**
	 * Create, remove or modify a tag naming a revision.
	 * @param tagName
	 * @param options see {@link ITagOptions}
	 * @throws BazaarClientException
	 */
	public void tag(String tagName, Option...options) throws BazaarClientException;

	/**
	 * List all tags
	 * @param options see {@link ITagsOptions}
	 * @return
	 * @throws BazaarClientException
	 */
	public List<IBazaarTag> tags(final Option... options) throws BazaarClientException;

	/**
	 * Set aside some changes from the current tree
	 *
	 * @param files
	 *            files to shelve
	 * @param options see {@link IShelveOptions}
	 * @throws BazaarClientException
	 */
	public void shelve(File[] files, Option... options) throws BazaarClientException;

	/**
	 * Restore shelved changes.
	 * @param shelfId
	 * 			shelf id to restore, null to restore most recently shelved changes.
	 * @param options see {@link IUnShelveOptions}
	 * @throws BazaarClientException
	 */
	public void unShelve(String shelfId, Option... options) throws BazaarClientException;

	/**
	 * List previously-shelved changes.
	 *
	 * @param options see {@link IShelveListOptions}
	 * @return 
	 * @throws BazaarClientException
	 */
	public List<IBazaarShelf> shelveList(Option... options) throws BazaarClientException;

	/**
	 * List files with conflicts.
	 *
	 * @param options see {@link IConflictsOptions}
	 * @return 
	 * @throws BazaarClientException
	 */
	public List<IBazaarConflict> conflicts(Option... options) throws BazaarClientException;

	/**
	 * Re-base a branch.
	 *
	 * @param otherBranch
	 * @param options see {@link IRebaseOptions}
	 * @throws BazaarClientException
	 */
	public void rebase(BranchLocation otherBranch, Option...options) throws BazaarClientException;

	/**
	 * Returns true if there are still revisions that need to be replayed
	 *
	 * @param options see {@link IRebaseToDoOptions}
	 * @throws BazaarClientException
	 */
	public boolean rebaseToDo(Option...options) throws BazaarClientException;

	/**
	 * Continue an interrupted rebase after resolving conflicts
	 *
	 * @param options see {@link IRebaseContinueOptions}
	 * @throws BazaarClientException
	 */
	public void rebaseContinue(Option...options) throws BazaarClientException;

	/**
	 * Abort an interrupted rebase
	 *
	 * @param options see {@link IRebaseAbortOptions}
	 * @throws BazaarClientException
	 */
	public void rebaseAbort(Option...options) throws BazaarClientException;

	/**
	 * see {@link #pull(BranchLocation, Option...)}
	 * @return 
	 */
	String pull(final BranchLocation location, IBazaarProgressListener listener, final Option... options) throws BazaarClientException;

	/**
	 * Remove unwanted files from working tree 
	 *
	 * @param options see {@link ICleanTreeOptions}
	 * @throws BazaarClientException
	 */
	public void cleanTree(Option...options) throws BazaarClientException;

	/**
	 * see {@link #update(File, Option...)}
	 */
	public String update(File file, IBazaarProgressListener listener, Option...options) throws BazaarClientException;

	/**
	 * see {@link #switchBranch(BranchLocation, Option...)}
	 */
	public void switchBranch(BranchLocation location, IBazaarProgressListener listener, Option...options) throws BazaarClientException;

	public String whoami(Option... options) throws BazaarClientException;

	public void whoami(String name, Option... options) throws BazaarClientException;

	public void reconfigure(BranchLocation location, IBazaarProgressListener listener, Option... options) throws BazaarClientException;
}
