/**
 * LICENCSE + COPYRIGHT
 */
package org.vcs.bazaar.client.commandline;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import org.vcs.bazaar.client.BazaarClient;
import org.vcs.bazaar.client.BazaarNotificationHandler;
import org.vcs.bazaar.client.BazaarRevision;
import org.vcs.bazaar.client.BazaarRevision.Prefix;
import org.vcs.bazaar.client.BazaarRevisionRange;
import org.vcs.bazaar.client.BazaarTreeStatus;
import org.vcs.bazaar.client.BazaarVersion;
import org.vcs.bazaar.client.BazaarVersionInfo;
import org.vcs.bazaar.client.IBazaarAnnotation;
import org.vcs.bazaar.client.IBazaarConflict;
import org.vcs.bazaar.client.IBazaarInfo;
import org.vcs.bazaar.client.IBazaarItemInfo;
import org.vcs.bazaar.client.IBazaarLogMessage;
import org.vcs.bazaar.client.IBazaarNotifyListener;
import org.vcs.bazaar.client.IBazaarProgressListener;
import org.vcs.bazaar.client.IBazaarRevisionSpec;
import org.vcs.bazaar.client.IBazaarShelf;
import org.vcs.bazaar.client.IBazaarTag;
import org.vcs.bazaar.client.IBzrLogMessageHandler;
import org.vcs.bazaar.client.IPlugin;
import org.vcs.bazaar.client.commandline.commands.Add;
import org.vcs.bazaar.client.commandline.commands.Annotate;
import org.vcs.bazaar.client.commandline.commands.Bind;
import org.vcs.bazaar.client.commandline.commands.Branch;
import org.vcs.bazaar.client.commandline.commands.Cat;
import org.vcs.bazaar.client.commandline.commands.CheckOut;
import org.vcs.bazaar.client.commandline.commands.CleanTree;
import org.vcs.bazaar.client.commandline.commands.Commit;
import org.vcs.bazaar.client.commandline.commands.Conflicts;
import org.vcs.bazaar.client.commandline.commands.Diff;
import org.vcs.bazaar.client.commandline.commands.FindMergeBase;
import org.vcs.bazaar.client.commandline.commands.Ignore;
import org.vcs.bazaar.client.commandline.commands.Info;
import org.vcs.bazaar.client.commandline.commands.Init;
import org.vcs.bazaar.client.commandline.commands.Log;
import org.vcs.bazaar.client.commandline.commands.Ls;
import org.vcs.bazaar.client.commandline.commands.Merge;
import org.vcs.bazaar.client.commandline.commands.Missing;
import org.vcs.bazaar.client.commandline.commands.Move;
import org.vcs.bazaar.client.commandline.commands.Nick;
import org.vcs.bazaar.client.commandline.commands.Plugins;
import org.vcs.bazaar.client.commandline.commands.Pull;
import org.vcs.bazaar.client.commandline.commands.Push;
import org.vcs.bazaar.client.commandline.commands.Rebase;
import org.vcs.bazaar.client.commandline.commands.RebaseAbort;
import org.vcs.bazaar.client.commandline.commands.RebaseContinue;
import org.vcs.bazaar.client.commandline.commands.RebaseToDo;
import org.vcs.bazaar.client.commandline.commands.Reconfigure;
import org.vcs.bazaar.client.commandline.commands.Remove;
import org.vcs.bazaar.client.commandline.commands.Resolve;
import org.vcs.bazaar.client.commandline.commands.Revert;
import org.vcs.bazaar.client.commandline.commands.RevisionInfo;
import org.vcs.bazaar.client.commandline.commands.Revno;
import org.vcs.bazaar.client.commandline.commands.Send;
import org.vcs.bazaar.client.commandline.commands.Shelve;
import org.vcs.bazaar.client.commandline.commands.ShelveList;
import org.vcs.bazaar.client.commandline.commands.Status;
import org.vcs.bazaar.client.commandline.commands.Switch;
import org.vcs.bazaar.client.commandline.commands.Tag;
import org.vcs.bazaar.client.commandline.commands.Tags;
import org.vcs.bazaar.client.commandline.commands.UnBind;
import org.vcs.bazaar.client.commandline.commands.UnCommit;
import org.vcs.bazaar.client.commandline.commands.UnShelve;
import org.vcs.bazaar.client.commandline.commands.Unknowns;
import org.vcs.bazaar.client.commandline.commands.Update;
import org.vcs.bazaar.client.commandline.commands.Version;
import org.vcs.bazaar.client.commandline.commands.VersionInfo;
import org.vcs.bazaar.client.commandline.commands.Whoami;
import org.vcs.bazaar.client.commandline.commands.options.KeywordOption;
import org.vcs.bazaar.client.commandline.commands.options.Option;
import org.vcs.bazaar.client.commandline.internal.Command;
import org.vcs.bazaar.client.commandline.internal.CommandRunner;
import org.vcs.bazaar.client.commandline.internal.ShellCommandRunner;
import org.vcs.bazaar.client.commandline.parser.XMLConflictsParser;
import org.vcs.bazaar.client.commandline.parser.XMLInfoParser;
import org.vcs.bazaar.client.commandline.parser.XMLLogParser;
import org.vcs.bazaar.client.commandline.parser.XMLLsParser;
import org.vcs.bazaar.client.commandline.parser.XMLMissingParser;
import org.vcs.bazaar.client.commandline.parser.XMLPluginParser;
import org.vcs.bazaar.client.commandline.parser.XMLShelveListParser;
import org.vcs.bazaar.client.commandline.parser.XMLStatusParser;
import org.vcs.bazaar.client.commandline.parser.XMLTagsParser;
import org.vcs.bazaar.client.commandline.parser.XMLVersionParser;
import org.vcs.bazaar.client.commandline.syntax.ILogOptions;
import org.vcs.bazaar.client.commandline.syntax.ILsOptions;
import org.vcs.bazaar.client.commandline.syntax.IVersionInfoOptions;
import org.vcs.bazaar.client.core.BazaarClientException;
import org.vcs.bazaar.client.core.BranchLocation;
import org.vcs.bazaar.client.xmlrpc.XmlRpcCommandException;

/**
 * <p>
 * High level interface to Bazaar Commands
 * </p>
 *
 * @author Guillermo Gonzalez
 *
 */
public class CommandLineClient extends BazaarClient {

	/** by default use  the ShellCommandRunner */
	final protected CommandLineNotificationHandler notificationHandler;

	/**
	 *
	 * @param notificationHandler
	 */
	public CommandLineClient(final CommandLineNotificationHandler notificationHandler) {
		super();
		this.notificationHandler = notificationHandler;
	}

	public void add(final File[] files, final Option... options) throws BazaarClientException {
		final Add cmd = new Add(workDir, files);
		runCommand(cmd, options);
	}

	public IBazaarAnnotation annotate(final File file, final Option... options) throws BazaarClientException {
		Annotate cmd = new Annotate(workDir, file);
		setOptions(cmd, options);
		run(cmd);
		return CommandLineAnnotation.getAnnotationFromXml(cmd.getStandardOutput());
	}

	public void branch(final BranchLocation fromLocation, final File toLocation, final IBazaarRevisionSpec revision, final Option... options) throws BazaarClientException {
		branch(fromLocation, toLocation, revision, null, options);
	}

	public InputStream cat(final File file, final IBazaarRevisionSpec revision, final String charsetName, final Option... options) throws BazaarClientException {
		final Cat cmd = new Cat(workDir, file);
		return cat(cmd, revision, charsetName, options);
	}

	private InputStream cat(final Cat cmd, final IBazaarRevisionSpec revision, final String charsetName, final Option... options) throws BazaarClientException {
		if (revision == null) {
			cmd.setOption(Cat.REVISION.with(BazaarRevision.getRevision(BazaarRevision.Prefix.LAST, "").toString()));
		} else {
			cmd.setOption(Cat.REVISION.with(revision.toString()));
		}
		setOptions(cmd, options);
		run(cmd);
		if(charsetName != null) {
			try {
				return new ByteArrayInputStream(cmd.getStandardOutput(charsetName).getBytes(charsetName));
			} catch (UnsupportedEncodingException e) {
				return new ByteArrayInputStream(cmd.getStandardOutput().getBytes());
			}
		} else {
			return new ByteArrayInputStream(cmd.getStandardOutput().getBytes());
		}
	}

	public void checkout(final BranchLocation fromLocation, final File toLocation, final Option... options) throws BazaarClientException {
		checkout(fromLocation, toLocation, null, options);
	}

	public void commit(final File[] files, String message, final Option... options) throws BazaarClientException {
		final Commit cmd = new Commit(workDir, files, message);
		runCommand(cmd, options);
	}

	public String diff(final File[] files, final IBazaarRevisionSpec range, final Option... options) throws BazaarClientException {
		final Diff cmd = new Diff(workDir, files);
		if (range != null) {
			cmd.setOption(Diff.REVISION.with(range.toString()));
		}
		setOptions(cmd, options);
		run(cmd);
		return cmd.getStandardOutput();
	}

	public void init(final File location, final Option... options) throws BazaarClientException {
		final Init cmd = new Init(location);
		setOptions(cmd, options);
		run(cmd);
	}

	public List<IBazaarLogMessage> log(final File location, final Option... options) throws BazaarClientException {
		final Log cmd = new Log(workDir, location);
		return log(cmd, options);
	}

	public List<IBazaarLogMessage> log(URI location, final Option... options) throws BazaarClientException {
		final Log cmd = new Log(workDir, location);
		return log(cmd, options);
	}

	public List<IBazaarLogMessage> log(final BranchLocation location, final Option... options) throws BazaarClientException {
		final Log cmd = new Log(workDir, location);
		return log(cmd, options);
	}

	/**
	 * This version of log, is optimized for BIG history fetching, it fetch the history in
	 * "groups" of 10000 logs and spawns a thread to parse and call the {@link IBzrLogMessageHandler#handle(IBazaarLogMessage)}.
	 */
	public void log(final BranchLocation location, final IBzrLogMessageHandler logHandler, Option... options) throws BazaarClientException {
		log(location, logHandler, false, options);
	}

	public void logAsync(final BranchLocation location, final IBzrLogMessageHandler logHandler, Option... options) throws BazaarClientException {
		log(location, logHandler, true, options);
	}

	protected void log(final BranchLocation location, final IBzrLogMessageHandler logHandler, final boolean async, Option...options) throws BazaarClientException {
		int limit = 1;
		int fetchSize = 1000;
		options = ILogOptions.REVISION.removeFrom(options);
		Arrays.sort(options);
		final int limitIdx = Arrays.binarySearch(options, ILogOptions.LIMIT);
		if(limitIdx >= 0) {
			final String limitStr = ((KeywordOption)options[limitIdx]).getArgument();
			if(limitStr != null && !"".equals(limitStr)) {
				limit = Integer.valueOf(limitStr);
			}
			options = ILogOptions.LIMIT.removeFrom(options);
		}
		final int verboseIdx = Arrays.binarySearch(options, ILogOptions.VERBOSE);
		if(verboseIdx >= 0) {
			fetchSize = 100;
		}
		final int iterations = limit/fetchSize;
		final ExecutorService executor = Executors.newFixedThreadPool(1);
		final List<Future<? extends Object>> futures = new ArrayList<Future<? extends Object>>(iterations);
		int startRev = fetchSize;
		int endRev = 1;
		BazaarRevision lastRevno = revno(location);
		int endRevMaxValue = Integer.valueOf(lastRevno.getValue());
		for (int i = 0; i < iterations; i++) {
			startRev = fetchSize*(i+1);
			if(startRev > endRevMaxValue) {
				break;
			}
			// create the command here because it keep state (options, output, etc)
			// FIXME: needs to be fixed in Command class
			final Log cmd = new Log(workDir, location);
			for (Option option : options) {
				cmd.setOption(option);
			}
			BazaarRevision start = BazaarRevision.getRevision(Prefix.LAST, String.valueOf(startRev));
			BazaarRevision end = BazaarRevision.getRevision(Prefix.LAST, String.valueOf(endRev));
			cmd.setOption(ILogOptions.REVISION.with(BazaarRevisionRange.getRange(start, end).toString()));
			run(cmd);
			final String xml = cmd.getStandardOutput();
			Runnable logHandlerRunnable = new Runnable() {
				public void run() {
					try {
						logHandler.handle(XMLLogParser.parse(xml));
					} catch (BazaarClientException e) {
						CommandLineClient.this.notificationHandler.logException(e);
					}
				}
			};
			futures.add(executor.submit(logHandlerRunnable));
			endRev = startRev+1;
			if(endRev > endRevMaxValue) {
				endRev = endRevMaxValue;
			}
		}
		if(!async) {
			for (Future<? extends Object> future : futures) {
				try {
					future.get();
				} catch (InterruptedException e) {
					executor.shutdownNow();
					throw BazaarClientException.wrapException(e);
				} catch (ExecutionException e) {
					executor.shutdownNow();
					throw BazaarClientException.wrapException(e);
				}
			}
		}
		executor.shutdown();
	}

	private List<IBazaarLogMessage> log(final Command cmd, final Option... options) throws BazaarClientException {
		setOptions(cmd, options);
		run(cmd);
		return XMLLogParser.parse(cmd.getStandardOutput());
	}

	public void move(final File[] orig, final File dest, final Option... options) throws BazaarClientException {
		final Move cmd = new Move(workDir, orig, dest);
		setOptions(cmd, options);
		run(cmd);
	}

	public String nick(final String newNick) throws BazaarClientException {
		final Nick cmd = new Nick(workDir, newNick);
		run(cmd);
		String nick = cmd.getStandardOutput();
		return nick==null?"":nick.replaceAll("\n", "").replaceAll("\r", "");
	}

	public Set<IPlugin> plugins() throws BazaarClientException {
	    final Plugins cmd = new Plugins();
        run(cmd);
	    return new XMLPluginParser().parse(cmd.getStandardOutput());
	}

	public String pull(final BranchLocation location, final Option... options) throws BazaarClientException {
		return pull(location, null, options);
	}

	public void push(final BranchLocation location, final Option... options) throws BazaarClientException {
		final Push cmd = new Push(workDir, location);
		runCommand(cmd, options);
	}

	public void remove(final File[] files, final Option... options) throws BazaarClientException {
		final Remove cmd = new Remove(workDir, files);
		setOptions(cmd, options);
		run(cmd);
	}

	public void revert(final File[] files, final Option... options) throws BazaarClientException {
		final Revert cmd = new Revert(workDir, files);
		setOptions(cmd, options);
		run(cmd);
	}

	public BazaarRevision revno(final BranchLocation location) throws BazaarClientException {
		Revno cmd;
		if (location != null)
			cmd = new Revno(workDir, location);
		else
			cmd = new Revno(workDir, new BranchLocation(new File(".")));
		run(cmd);
		return BazaarRevision.getRevision(BazaarRevision.Prefix.REVNO, cmd.getStandardOutput());
	}

	public BazaarRevision revisionInfo(final File location, final IBazaarRevisionSpec revision) throws BazaarClientException {
		RevisionInfo cmd;
		if (location != null)
			cmd = new RevisionInfo(location);
		else
			cmd = new RevisionInfo(workDir);

		if(revision != null)
			cmd.setOption(RevisionInfo.REVISION.with(revision.toString()));

		run(cmd);
		String revString = cmd.getStandardOutput();
		revString = revString.substring(revString.lastIndexOf(' ') + 1).trim();
		return BazaarRevision.getRevision(Prefix.REVID, revString);
	}

	public BazaarTreeStatus status(File[] files, final Option... options) throws BazaarClientException {

		if(files == null) {
			files = new File[]{ new File(".") };
		}

		final Status cmd = new Status(workDir, files);

		setOptions(cmd, options);

		run(cmd);

		BazaarTreeStatus status = null;
		if(cmd.getStandardOutput() != null) {
			final XMLStatusParser parser = new XMLStatusParser();
			parser.parse(cmd.getStandardOutput());
			status = new BazaarTreeStatus(parser.getStatusSet(), parser.getPendingMerges());
		} else {
			status = new BazaarTreeStatus();
		}
		return status;
	}

	public void unCommit(final File location, final Option... options) throws BazaarClientException {
		final UnCommit cmd = new UnCommit(workDir, location);
		// use --force because (for the moment) we can't propmt a comfirm dialog
		cmd.setOption(UnCommit.FORCE);
		runCommand(cmd, options);
	}

	public void ignore(final File file, final String pattern) throws BazaarClientException {
		final Ignore cmd = new Ignore(file, pattern);
		run(cmd);
	}

	public Map<String, String> ignored(final File file) throws BazaarClientException {
		IBazaarItemInfo[] ignored = this.ls(file, null, Ls.IGNORED);
		Map<String, String> result = new HashMap<String, String>(ignored.length);
		for (IBazaarItemInfo item : ignored) {
			result.put(item.getPath(), item.getPattern());
		}
		return result;
	}

	public String[] unknowns(final File workDir) throws BazaarClientException {
		final Unknowns cmd = new Unknowns(workDir, null);
		run(cmd);
		return cmd.getStandardOutputSplit();
	}

	public IBazaarItemInfo[] ls(final File workDir, final IBazaarRevisionSpec revision, final Option... options) throws BazaarClientException {
		final Ls cmd = new Ls(workDir);
		return ls(cmd, revision, options);
	}

	public IBazaarItemInfo[] ls(final BranchLocation location, final IBazaarRevisionSpec revision, final Option... options) throws BazaarClientException {
		final Ls cmd = new Ls(workDir, location);
		return ls(cmd, revision, options);
	}

	public IBazaarItemInfo[] ls(final URI location, final IBazaarRevisionSpec revision, final Option... options) throws BazaarClientException {
		final Ls cmd = new Ls(workDir, location);
		return ls(cmd, revision, options);
	}

	private IBazaarItemInfo[] ls(Ls cmd, final IBazaarRevisionSpec revision, final Option... options) throws BazaarClientException {
		if (revision != null) {
			cmd.setOption(ILsOptions.REVISION.with(revision.toString()));
		}
		setOptions(cmd, options);
		try {
			run(cmd);
			return XMLLsParser.parse(cmd.getStandardOutput()).toArray(new IBazaarItemInfo[0]);
		} catch (BazaarClientException e) {
			// if get an exception from 'bzr ls', assume that the
			// resource is not under revision control
			return new IBazaarItemInfo[0];
		}
	}

	public String update(final File file, final Option...options) throws BazaarClientException {
		return update(file, null, options);
	}

	public IBazaarInfo info(final BranchLocation location, final Option... options) throws BazaarClientException {
		final Info cmd = new Info(workDir, location);
		setOptions(cmd, options);
		run(cmd);
		IBazaarInfo info = new XMLInfoParser().parse(cmd.getStandardOutput());
		if(info == null) { 
			// something bad happened, bail out!
			throw new BazaarClientException("An unexpected error occurred while getting the info of: " + location.toString());
		}
		return info;
	}

	public void bind(final BranchLocation location, final Option...options) throws BazaarClientException {
		final Bind cmd = new Bind(workDir, location);
		runCommand(cmd, options);
	}

	public void unBind(final Option...options) throws BazaarClientException {
		final UnBind cmd = new UnBind(workDir);
		runCommand(cmd, options);
	}

	public Map<String, List<IBazaarLogMessage>> missing(final File workdir, final BranchLocation otherBranch, final Option... options) throws BazaarClientException {
		final Missing cmd = new Missing(workDir, otherBranch);
		setOptions(cmd, options);
		run(cmd);
		if(cmd.getStandardOutput() == null || cmd.getStandardOutput().trim().equals(""))
			return new HashMap<String, List<IBazaarLogMessage>>(0);
		return new XMLMissingParser().parse(cmd.getStandardOutput());
	}

	public void switchBranch(final BranchLocation location, final Option... options) throws BazaarClientException {
		switchBranch(location, null, options);
	}

	public void merge(final BranchLocation remoteBranch, final Option... options) throws BazaarClientException {
		final Merge cmd = new Merge(workDir, remoteBranch);
		runCommand(cmd, options);
	}

	public void resolve(List<File> files, final Option...options) throws BazaarClientException {
		final Resolve cmd = new Resolve(workDir, files);
		setOptions(cmd, options);
		run(cmd);
	}

	public void send(final BranchLocation submitBranch, final Option... options) throws BazaarClientException {
		final Send cmd = new Send(workDir, submitBranch);
		setOptions(cmd, options);
		run(cmd);
	}

	public BazaarVersionInfo versionInfo(final BranchLocation location, final Option... options)
			throws BazaarClientException {
		final VersionInfo cmd = new VersionInfo(workDir, location);
		setOptions(cmd, options);
		cmd.setOption(IVersionInfoOptions.FORMAT.with("rio"));
		run(cmd);
		return BazaarVersionInfo.parse(cmd.getStandardOutput());
	}

    public BazaarVersion version(final Option... options)
            throws BazaarClientException {
       final Version cmd = new Version();
       setOptions(cmd, options);
       run(cmd);
       return XMLVersionParser.parse(cmd.getStandardOutput());
    }

	public void addNotifyListener(IBazaarNotifyListener listener) {
		notificationHandler.add(listener);
	}

	public BazaarNotificationHandler getNotificationHandler() {
		return notificationHandler;
	}

	public void removeNotifyListener(IBazaarNotifyListener listener) {
		notificationHandler.remove(listener);
	}

	/**
	 * Run the specified command.<br>
	 * All output is reported to the client {@link BazaarNotificationHandler}
	 *
	 * @param cmd ICommand to execute
	 * @throws BazaarClientException
	 */
	protected void run(final Command cmd) throws BazaarClientException {
		run(cmd, getCommandRunner());
	}

	protected void run(final Command cmd, final CommandRunner runner) throws BazaarClientException {
		try {
			notificationHandler.logCommandLine(getClass().getSimpleName() + " " + cmd.getCommandInfo());
			runner.setPasswordCallback(userPasswordPrompt);
			cmd.execute(runner);
			final String err = cmd.getStandardError();
			if(err != null && err.length() > 0)
				notificationHandler.logError(cmd.getStandardError());
		} catch (BazaarClientException e) {
			notificationHandler.logError(cmd.getCommandError());
			notificationHandler.logException(e);
			throw e;
		}
	}

	public BazaarRevision findMergeBase(final BranchLocation branch, final BranchLocation other)
		throws BazaarClientException {
		final FindMergeBase cmd = new FindMergeBase(branch, other);
		run(cmd);
		String output = cmd.getStandardOutput();
		output = output.replace("merge base is revision ", "").trim();
		return BazaarRevision.getRevision(Prefix.REVID, output);
	}

	@Override
	protected CommandRunner getCommandRunner() {
		return new ShellCommandRunner();
	}

	public void tag(String tagName, Option... options) throws BazaarClientException {
		final Tag cmd = new Tag(workDir, tagName);
		runCommand(cmd, options);
	}

	public List<IBazaarTag> tags(final Option... options) throws BazaarClientException {
		final Tags cmd = new Tags(workDir);
		setOptions(cmd, options);
		run(cmd);
		return XMLTagsParser.parse(cmd.getStandardOutput());
	}

	public void shelve(File[] files, Option... options) throws BazaarClientException {
		final Shelve cmd = new Shelve(workDir, files);
		runCommand(cmd, options);
	}

	public void unShelve(String shelfId, Option... options) throws BazaarClientException {
		final UnShelve cmd = new UnShelve(workDir, shelfId);
		runCommand(cmd, options);
	}

	public List<IBazaarShelf> shelveList(Option... options) throws BazaarClientException {
		final ShelveList cmd = new ShelveList(workDir);
		setOptions(cmd, options);
		run(cmd);
		return XMLShelveListParser.parse(cmd.getStandardOutput());
	}

	public List<IBazaarConflict> conflicts(Option... options) throws BazaarClientException {
		final Conflicts cmd = new Conflicts(workDir);
		setOptions(cmd, options);
		run(cmd);
		return XMLConflictsParser.parse(cmd.getStandardOutput());
	}

	public InputStream cat(BranchLocation location, IBazaarRevisionSpec revision, final String charsetName,  Option... options)
			throws BazaarClientException {
		final Cat cmd = new Cat(workDir, location);
		return cat(cmd, revision, charsetName, options);
	}

	public InputStream cat(URI location, IBazaarRevisionSpec revision, final String charsetName, Option... options) throws BazaarClientException {
		final Cat cmd = new Cat(workDir, location);
		return cat(cmd, revision, charsetName, options);
	}

	public void branch(BranchLocation fromLocation, File toLocation, IBazaarRevisionSpec revision, 
						IBazaarProgressListener listener, Option... options) throws BazaarClientException {
		final Branch cmd = new Branch(fromLocation, toLocation);
		cmd.setProgressListener(listener);
		if (revision != null)
			cmd.setOption(Branch.REVISION.with(revision.toString()));
		runCommand(cmd, options);
	}

	public void checkout(BranchLocation fromLocation, File toLocation, IBazaarProgressListener listener,
						Option... options) throws BazaarClientException {
		final CheckOut cmd = new CheckOut(fromLocation, toLocation);
		cmd.setProgressListener(listener);
		runCommand(cmd, options);
	}

	public void rebase(final BranchLocation remoteBranch, final Option... options) throws BazaarClientException {
		final Rebase cmd = new Rebase(workDir, remoteBranch);
		runCommand(cmd, options);
	}

	public boolean rebaseToDo(final Option... options) throws BazaarClientException {
		final RebaseToDo cmd = new RebaseToDo(workDir);
		setOptions(cmd, options);
		try {
			run(cmd);
		} catch (XmlRpcCommandException e) {
			if ("No rebase in progress".equals(e.getMessage())) {
				return false;
			}
			throw e;
		}
		notificationHandler.logCompleted(cmd.getStandardOutput());
		return true;
	}

	public void rebaseContinue(Option... options) throws BazaarClientException {
		RebaseContinue cmd = new RebaseContinue(workDir);
		runCommand(cmd, options);
	}

	public void rebaseAbort(Option... options) throws BazaarClientException {
		RebaseAbort cmd = new RebaseAbort(workDir);
		runCommand(cmd, options);
	}

	public String pull(final BranchLocation location, IBazaarProgressListener listener, final Option... options)
			throws BazaarClientException {
		final Pull cmd = new Pull(workDir, location);
		cmd.setProgressListener(listener);
		runCommand(cmd, options);
		return cmd.getStandardError();
	}

	private void runCommand(Command cmd, Option... options) throws BazaarClientException {
		setOptions(cmd, options);
		run(cmd);
		notificationHandler.logCompleted(cmd.getStandardOutput());
	}

	public void setOptions(final Command cmd, final Option... options) {
		for (Option option : options) {
			cmd.setOption(option);
		}
	}

	public void cleanTree(Option... options) throws BazaarClientException {
		final CleanTree cmd = new CleanTree(workDir);
		runCommand(cmd, options);
	}

	public String update(final File file, IBazaarProgressListener listener, final Option...options) throws BazaarClientException {
		final Update cmd = new Update(workDir, file);
		cmd.setProgressListener(listener);
		setOptions(cmd, options);
		run(cmd);
		return cmd.getStandardError();
	}

	public void switchBranch(final BranchLocation location, IBazaarProgressListener listener, final Option... options) throws BazaarClientException {
		final Switch cmd = new Switch(workDir, location);
		cmd.setProgressListener(listener);
		runCommand(cmd, options);
	}

	public String whoami(Option... options) throws BazaarClientException {
		final Whoami cmd = new Whoami(workDir);
		runCommandWithShellCommandRunner(cmd, options);
		return cmd.getStandardOutput().trim();
	}

	public void whoami(String name, Option... options) throws BazaarClientException {
		final Whoami cmd = new Whoami(workDir, name);
		runCommandWithShellCommandRunner(cmd, options);
	}

	private void runCommandWithShellCommandRunner(Whoami cmd, Option[] options) throws BazaarClientException {
		setOptions(cmd, options);
		run(cmd, new ShellCommandRunner());
		notificationHandler.logCompleted(cmd.getStandardOutput());
	}

	public void reconfigure(BranchLocation location, IBazaarProgressListener listener, Option... options) throws BazaarClientException {
		Reconfigure cmd = new Reconfigure(workDir, location);
		cmd.setProgressListener(listener);
		runCommand(cmd, options);
	}

}
