/**
 * LICENSE + COPYRIGHT
 */
package org.vcs.bazaar.client.commandline.parser;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.kxml2.io.KXmlParser;
import org.vcs.bazaar.client.IBazaarLogMessage;
import org.vcs.bazaar.client.IBazaarStatus;
import org.vcs.bazaar.client.commandline.CommandLineLogMessage;
import org.vcs.bazaar.client.core.BazaarClientException;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

/**
 * I'm a parser for the output generated by the command: log --xml (using
 * bzr-xmloutput plugin), and with that output I create a
 * Set&lt;IBazaarLogMessage&gt;
 *
 * @author Guillermo Gonzalez
 */
public class XMLLogParser extends XMLParser {

	public static final String REVNO = "revno";
	public static final String REVISION_ID = "revisionid";
	public static final String COMMITER = "committer";
	public static final String BRANCH_NICK = "branch-nick";
	public static final String TIMESTAMP = "timestamp";
	public static final String MESSAGE = "message";
	public static final String LOG = "log";
	public static final String TAGS = "tags";
	public static final String TAG = "tag";
	public static final String PARENTS = "parents";
	public static final String PARENT = "parent";
	public static final String MERGE = "merge";
	public static final String FILES = "affected-files";

	/**
	 *
	 * @param xml a String containing the xml to be parsed
	 * @return List<IBazaarLogMessage> a unmodifiable and ordered (by revno) list of log msgs
	 * @throws BazaarClientException
	 */
	public static List<IBazaarLogMessage> parse(String xml) throws BazaarClientException {
		return parse(new StringReader(xml));
	}

	/**
	 *
	 * @param aReader a reader for the xml input
	 * @return List<IBazaarLogMessage> a unmodifiable and ordered (by revno) list of log msgs
	 * @throws BazaarClientException
	 */
	public static List<IBazaarLogMessage> parse(Reader aReader) throws BazaarClientException {
		KXmlParser parser = new KXmlParser();
		try {
			parser.setInput(aReader);
			return new XMLLogParser().parse(parser);
		} catch (XmlPullParserException e) {
			throw BazaarClientException.wrapException(e);
		}
	}

	protected List<IBazaarLogMessage> parse(KXmlParser aParser) throws BazaarClientException {
		// more efficient to reference a stack variable(within a method) instead of a class variable. everytime you do an add etc.
		final List<IBazaarLogMessage> logs = new ArrayList<IBazaarLogMessage>();
		this.parser = aParser;
		try {
			int eventType = parser.getEventType();
			// iterate over all tags (actually only care about first level <log/> tags)
			while (eventType != XmlPullParser.END_DOCUMENT) {
				if (eventType == XmlPullParser.START_TAG && LOG.equals(parser.getName())) {
					IBazaarLogMessage log = parseLog();
					if (log != null) {
						logs.add(log);
					}
				}
				eventType = parser.next();
			}
		} catch (XmlPullParserException e) {
			throw BazaarClientException.wrapException(e);
		} catch (IOException e) {
			throw BazaarClientException.wrapException(e);
		}
		Collections.sort(logs, new IBazaarLogMessage.LogMessageComparator());
		return logs;
	}

	/**
	 * This method parse one <log>...</log> including <merge/>
	 * @throws IOException
	 * @throws XmlPullParserException
	 *
	 */
	protected CommandLineLogMessage parseLog() throws XmlPullParserException, IOException {
		final XMLStatusParser statusParser = new XMLStatusParser();
		String revno, commiter, nick, timestamp, message, revisionId;
		revno = commiter = nick = timestamp = message = revisionId = null;
		List<String> parents = null;
		List<IBazaarStatus> statuses = null;
		final List<IBazaarLogMessage> mergedLogs = new ArrayList<IBazaarLogMessage>();
		List<String> tags = null;
		int eventType = parser.next();
		while (!(eventType == XmlPullParser.END_TAG && LOG.equals(parser.getName())) || eventType == XmlPullParser.END_DOCUMENT) {
			if (eventType == XmlPullParser.START_TAG && REVNO.equals( parser.getName() )) {
				revno = parser.nextText();
			} else if (eventType == XmlPullParser.START_TAG && REVISION_ID.equals( parser.getName() )) {
				revisionId = parser.nextText();
			} else if (eventType == XmlPullParser.START_TAG && PARENTS.equals( parser.getName() )) {
				parents = parseParents();
			} else if (eventType == XmlPullParser.START_TAG && TAGS.equals( parser.getName() )) {
				tags = parseTags();
			} else if (eventType == XmlPullParser.START_TAG && COMMITER.equals( parser.getName() )) {
				commiter = parser.nextText();
			} else if (eventType == XmlPullParser.START_TAG && BRANCH_NICK.equals( parser.getName() )) {
				nick = parser.nextText();
			} else if (eventType == XmlPullParser.START_TAG && TIMESTAMP.equals( parser.getName() )) {
				timestamp = parser.nextText();
			} else if (eventType == XmlPullParser.START_TAG && MESSAGE.equals( parser.getName() )) {
				// FIXME: for some reason \b and \t are not correctly handled in client <-> server
				final String rawMsg = parser.nextText();
				message = rawMsg!=null?rawMsg.replace("\\x0c", "\f").replace("\\x09", "\t").replace("\\x08", "\b"):"(no message)";
			} else if (eventType == XmlPullParser.START_TAG && MERGE.equals( parser.getName() )) {
				mergedLogs.addAll(parseMergedLogs());
			} else if (eventType == XmlPullParser.START_TAG && FILES.equals( parser.getName() )) {
				statuses = statusParser.parseForLog(parser, FILES);
			}
			eventType = parser.next();
		}
		return new CommandLineLogMessage(revno, commiter, null, nick, timestamp, message,
				statuses, mergedLogs, revisionId, parents, tags);
	}

	private List<IBazaarLogMessage> parseMergedLogs() throws XmlPullParserException, IOException {
		List<IBazaarLogMessage> logs = new ArrayList<IBazaarLogMessage>(0);
		int eventType = parser.next();
		int mergeCounter = 1;
		while (mergeCounter > 0) {
			if (eventType == XmlPullParser.START_TAG && MERGE.equals(parser.getName())) {
				mergeCounter++;
			} else if (eventType == XmlPullParser.START_TAG && LOG.equals(parser.getName())) {
				logs.add(parseLog());
			}
			eventType = parser.next();
			if(eventType == XmlPullParser.END_TAG && MERGE.equals(parser.getName())) {
				mergeCounter--;
			}
		}
		Collections.sort(logs, new IBazaarLogMessage.LogMessageComparator());
		return logs;
	}

	private List<String> parseParents() throws XmlPullParserException, IOException {
		final List<String> parents = new ArrayList<String>(0);
		int eventType = parser.next();
		while (eventType != XmlPullParser.END_TAG && !PARENTS.equals(parser.getName())) {
			if (eventType == XmlPullParser.START_TAG && PARENT.equals(parser.getName())) {
				parents.add(parser.nextText());
			}
			eventType = parser.next();
		}
		return parents;
	}

	private List<String> parseTags() throws XmlPullParserException, IOException {
		final List<String> tags = new ArrayList<String>(0);
		int eventType = parser.next();
		while (eventType != XmlPullParser.END_TAG && !TAGS.equals(parser.getName())) {
			if (eventType == XmlPullParser.START_TAG && TAG.equals(parser.getName())) {
				tags.add(parser.nextText());
			}
			eventType = parser.next();
		}
		return tags;
	}
}
