/**
 *
 */
package org.vcs.bazaar.client.xmlrpc.internal;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.util.List;

import org.vcs.bazaar.client.BazaarClientPreferences;
import org.vcs.bazaar.client.IBazaarProgressListener;
import org.vcs.bazaar.client.IBazaarUserPassword;
import org.vcs.bazaar.client.commandline.CommandLineException;
import org.vcs.bazaar.client.commandline.internal.Command;
import org.vcs.bazaar.client.commandline.internal.CommandRunner;
import org.vcs.bazaar.client.commandline.internal.IProgressListenerAware;
import org.vcs.bazaar.client.core.BazaarClientException;
import org.vcs.bazaar.client.core.BranchLocation;
import org.vcs.bazaar.client.xmlrpc.BzrXmlRpcError;
import org.vcs.bazaar.client.xmlrpc.XmlRpcCommandException;
import org.vcs.bazaar.client.xmlrpc.XmlRpcMethod;

import redstone.xmlrpc.XmlRpcArray;
import redstone.xmlrpc.XmlRpcException;
import redstone.xmlrpc.XmlRpcFault;

/**
 * @author Guillermo Gonzalez <guillo.gonzo AT gmail DOT com>
 * 
 */
public class XMLRPCCommandRunner extends CommandRunner {

	private final XMLRPCServer server;
	protected XmlRpcArray result;

	public XMLRPCCommandRunner(XMLRPCServer server) {
		this(server, BazaarClientPreferences.getInstance());
	}

	public XMLRPCCommandRunner(XMLRPCServer server, BazaarClientPreferences preferences) {
		super(preferences);
		this.server = server;
	}
	
	private XmlRpcArray runCommand(final Command command, final boolean checkExitValue, final File workDir)
			throws XmlRpcCommandException {
		XMLRPCProgressMonitor monitor = null;
		List<String> cmdLine;
		try {
			cmdLine = command.constructCommandInvocationString();
		} catch (CommandLineException e) {
			throw XmlRpcCommandException.wrapException((Exception) e);
		}
		if (command instanceof IProgressListenerAware
				&& ((IProgressListenerAware) command).getProgressListener() != null) {
			monitor = new XMLRPCProgressMonitor(cmdLine, ((IProgressListenerAware) command).getProgressListener());
			monitor.start();
		}
		cmdLine.addAll(0, BazaarClientPreferences.getExecutable(false));
		try {
			server.startCommand();
			try {
				Object[] params = new Object[] {cmdLine,
						workDir != null ? workDir.getPath() : new File(".").getCanonicalPath()};
				return runCommand(checkExitValue, "run_bzr", params);
			} catch (XmlRpcCommandException e) {
				return handleAuthorizationErrors(checkExitValue, workDir, cmdLine, e);
			}
		} catch (IOException e) {
			throw XmlRpcCommandException.wrapException(e);
		} finally {
			server.endCommand();
			if (monitor != null) {
				monitor.stop();
			}
		}
	}

	private XmlRpcArray handleAuthorizationErrors(final boolean checkExitValue, final File workDir, List<String> cmdLine,
			XmlRpcCommandException e) throws IOException, XmlRpcCommandException {
		String msg = e.getMessage();
		if (msg != null && msg.startsWith("XMLRPC authentication")) {
			if (userPasswordPrompt != null) {
				String prompt = getUserPasswordPromptMessage(cmdLine, msg);
				boolean passwordOnly = e.getMessage().startsWith("XMLRPC authentication password");
				IBazaarUserPassword credentials = userPasswordPrompt.getCredentials(prompt, passwordOnly, false);
				if (credentials != null) {
					String user = credentials.getUser() != null ? credentials.getUser() : "";
					String password = credentials.getPassword() != null ? credentials.getPassword() : "";
					Object[] params = new Object[] { cmdLine,
							workDir != null ? workDir.getPath() : new File(".").getCanonicalPath(), user,
							password };
					try {
						return runCommand(checkExitValue, "run_bzr_auth", params);
					} catch (XmlRpcCommandException e1) {
						/* if authentication with stored credentials failed try to ask for new ones */
						if (credentials.isStored()) {
							credentials = userPasswordPrompt.getCredentials(prompt, passwordOnly, true);
							if (credentials != null) {
								user = credentials.getUser() != null ? credentials.getUser() : "";
								password = credentials.getPassword() != null ? credentials.getPassword() : "";
								params = new Object[] { cmdLine,
										workDir != null ? workDir.getPath() : new File(".").getCanonicalPath(),
										user, password };
								return runCommand(checkExitValue, "run_bzr_auth", params);
							}
						} else {
							throw e1;
						}
					}
				}
			}
		}
		throw e;
	}

	private String getUserPasswordPromptMessage(List<String> cmdLine, String msg) {
		/* default bzr password prompt */
		int index = msg.indexOf(':');
		String prompt = "";
		if (index >= 0 && index + 1 < msg.length()) {
			prompt = msg.substring(index);
		}
		/* let's try to get the location from command line arguments */
		if (cmdLine.size() > 1) {
			String branchLocation = cmdLine.get(cmdLine.size() - 1);
			if (cmdLine.size() > 2
					&& (cmdLine.get(1).equals("branch") || cmdLine.get(1).equals("checkout"))) {
				branchLocation = cmdLine.get(2);
			}
			try {
				BranchLocation location = new BranchLocation(branchLocation);
				prompt = location.toString();
			} catch (BazaarClientException e1) {
			}
		}
		return prompt;
	}

	private XmlRpcArray runCommand(final boolean checkExitValue, String methodName, final Object[] params)
			throws XmlRpcCommandException {
		try {
			try {
				final XmlRpcArray result = (XmlRpcArray) server.getClient(false).invoke(methodName, params);
				if (checkExitValue && result.getInteger(0) != 0 && !"".equals(result.getString(2))) {
					final XmlRpcCommandException e = new XmlRpcCommandException(BzrXmlRpcError.fromXml(result
							.getString(2)));
					e.setStackTrace(XmlRpcCommandException.getCurrentStackTrace());
					throw e;
				}
				return result;
			} catch (XmlRpcException e) {
				throw e;
			}
		} catch (XmlRpcFault fault) {
			XmlRpcCommandException e = null;
			try {
				e = new XmlRpcCommandException(BzrXmlRpcError.fromXml(fault.getMessage()));
				e.setStackTrace(BazaarClientException.getCurrentStackTrace());
			} catch (Exception e1) {
				// Error while parsing the xml error
				e = XmlRpcCommandException.wrapException(fault);
			}
			throw e;
		} catch (MalformedURLException e) {
			throw XmlRpcCommandException.wrapException(e);
		}
	}

	/**
	 * Executes a xmlrpc method.
	 * 
	 * @param method
	 *            the method's name
	 * @param params
	 *            a {@link XmlRpcArray} representing the method arguments
	 * @return the result of executing the method
	 * @throws XmlRpcCommandException
	 */
	Object executeMethod(String method, XmlRpcArray params) throws XmlRpcCommandException {
		try {
			return server.getClient(false).invoke(method, params);
		} catch (XmlRpcFault e) {
			throw XmlRpcCommandException.wrapException(e);
		} catch (MalformedURLException e) {
			throw XmlRpcCommandException.wrapException(e);
		}
	}

	@Override
	protected String getSplitExpression() {
		return "\n";
	}

	@Override
	public void runCommand(final Command command, final File workDir) throws BazaarClientException {
		result = new SafeXmlRPCCall<XmlRpcArray>() {
			@Override
			public XmlRpcArray internalExecute() throws XmlRpcCommandException {
				return runCommand(command, command.isCheckExitValue(), workDir);
			}
		}.execute();
	}

	/**
	 * Execute a xmlrpc method, but prior checks that the service is actually
	 * running by calling {@link #checkServiceStatus()}
	 * 
	 * @param method
	 *            a {@link XmlRpcMethod}
	 * @return
	 * @throws BazaarClientException
	 */
	@SuppressWarnings("unchecked")
	public Object executeMethod(final XmlRpcMethod method) throws BazaarClientException {
		final XmlRpcArray arguments = new XmlRpcArray();
		for (Object arg : method.getArguments()) {
			arguments.add(arg);
		}
		return new SafeXmlRPCCall<Object>() {
			@Override
			public Object internalExecute() throws XmlRpcCommandException {
				return executeMethod(method.getName(), arguments);
			}
		}.execute();
	}

	@Override
	public String getStandardOutput() {
		return decodeXMLOutput(result.getBinary(1));
	}

	@Override
	public String getStandardOutput(final String charsetName) throws UnsupportedEncodingException {
		return new String(result.getBinary(1), charsetName);
	}

	@Override
	public String getStandardError() {
		// No need to decode any bytes. If the string was decoded erroneously by
		// XML-RPC library,
		// we can't do anything as we don't have the original bytes to decode
		// properly, and we can't assume
		// string.getBytes() will give them to us either, as there's no
		// guarantee that they were originally
		// decoded from default system encoding (so that getBytes() would become
		// correct).
		return result.getString(2);
	}

	protected abstract class SafeXmlRPCCall<T> {

		public T execute() throws XmlRpcCommandException {
			try {
				return internalExecute();
			} catch (XmlRpcException e) {
				// if a XmlRpcException occurs, maybe the service is down
				// try one more time, but now check the service status
				// @note: this is ugly, but this way we avoid unneeded calls to
				// checkServiceStatus()
				// TODO: there should be a better way to do this.
				server.checkStatus();
				try {
					return internalExecute();
				} catch (XmlRpcException e1) {
					// if we get a second exception, something is *really* wrong
					throw XmlRpcCommandException.wrapException(e1);
				}
			}
		}

		public abstract T internalExecute() throws XmlRpcCommandException;

	}

	private class XMLRPCProgressMonitor implements Runnable {

		private boolean running;
		private List<String> cmdLine;
		private final IBazaarProgressListener listener;

		public XMLRPCProgressMonitor(List<String> cmdLine, IBazaarProgressListener listener) {
			this.listener = listener;
			this.cmdLine = cmdLine;
		}

		public void run() {
			while (running) {
				try {
					Thread.sleep(250);
				} catch (InterruptedException e) {
				}
				if (running) {
					try {
						final redstone.xmlrpc.XmlRpcClient client = server.getClient(true);
						Object[] params = new Object[] { cmdLine };
						final String result = (String) client.invoke("get_bzr_progress", params);
						if (result != null) {
							String progress = result;
							listener.logProgress(progress);
						}
					} catch (Exception e) {
						running = false;
					}
				}
			}
		}

		public void start() {
			running = true;
			Thread t = new Thread(this);
			t.setDaemon(true);
			t.start();
		}

		public void stop() {
			running = false;
		}
	}


}
