/**
 * LICENSE + COPYRIGHT
 */
package org.vcs.bazaar.client.utils;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.List;

import org.junit.Assert;
import org.junit.Test;

/**
 * @author Guillermo Gonzalez <guillo.gonzo AT gmail DOT com>
 *
 */
public class BazaarUtilitiesTest {

	@Test public void testGetRelativeUnix() {
		File base = new File("/home/user/bzr/project/trunk");
		File file = new File("/home/user/bzr/project/trunk/src_folder/file");
		File relative = BazaarUtilities.getRelativeTo(base, file);
		Assert.assertEquals(new File("src_folder/file"), relative);
	}

	@Test public void testGetRelativeWindows() {
		File base = new File("c:\\bzr\\project\\trunk");
		File file = new File("c:\\bzr\\project\\trunk\\src_folder\\file");
		File relative = BazaarUtilities.getRelativeTo(base, file);
		if(File.separatorChar == '\\')
			Assert.assertEquals(new File("src_folder\\file"), relative);
		else
			Assert.assertEquals(new File("src_folder/file"), relative);
	}

	@Test public void testUriEscapesMetaChar() {
		File base = new File("/home/file%07with%08funkyescapes");
		String uri = base.toURI().toString();
		Assert.assertEquals("file%2507with%2508funkyescapes",
							uri.substring(uri.lastIndexOf('/') + 1));

	}

	@Test public void testGetWindowsExecutable() throws IOException {
		File bat = File.createTempFile("bzr-eclipse", ".bat");
		try{
			FileWriter fw = new FileWriter(bat);
			fw.append("@C:\\\\Python25\\\\python.exe C:\\\\Python25\\\\Scripts\\\\bzr %*");
			fw.close();
			List<String> exe = BazaarUtilities.getWindowsExecutable(bat.getAbsolutePath());
			String[] expected = new String[]{"C:\\\\Python25\\\\python.exe", "C:\\\\Python25\\\\Scripts\\\\bzr"};
			Assert.assertArrayEquals(expected, exe.toArray(new String[0]));
		} finally{
			bat.delete();
		}
	}

	@Test
	public void testFindExecutable() throws IOException {
		String sysPath = System.getenv("PATH");
		File tmpFile = File.createTempFile("bzr-eclipse", "");
		tmpFile.delete();
		tmpFile.mkdirs();
		// create a file and insert it parent in PATH
		sysPath = tmpFile + File.pathSeparator + sysPath;
		// create a fake bzr executable
		File fakeBzr = new File(tmpFile, "bzr");
		if(BazaarUtilities.isWindows()) {
			fakeBzr = new File(tmpFile, "bzr.exe");
		}
		try {
			fakeBzr.createNewFile();
			Assert.assertEquals(fakeBzr, BazaarUtilities.findExecutable(sysPath));
			if(BazaarUtilities.isWindows()) {
				// if it's windows check for bzr.bat too
				fakeBzr.delete();
				fakeBzr = new File(tmpFile, "bzr.bat");
				fakeBzr.createNewFile();
				Assert.assertEquals(fakeBzr, BazaarUtilities.findExecutable(sysPath));
			}
		} finally {
			fakeBzr.delete();
			tmpFile.delete();
		}
	}

}
