/**
 * LICENCSE + COPYRIGHT
 */
package org.vcs.bazaar.client;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNotSame;
import static org.vcs.bazaar.client.testUtils.FileUtils.addContentToFile;
import static org.vcs.bazaar.client.testUtils.FileUtils.assertWTEqual;
import static org.vcs.bazaar.client.utils.BazaarUtilities.unixFilePath;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;

import org.junit.Assert;
import org.junit.Assume;
import org.junit.Ignore;
import org.junit.Test;
import org.vcs.bazaar.client.commandline.CommandLineClient;
import org.vcs.bazaar.client.commandline.CommandLineClientFactory;
import org.vcs.bazaar.client.commandline.syntax.ICommitOptions;
import org.vcs.bazaar.client.commandline.syntax.IInfoOptions;
import org.vcs.bazaar.client.commandline.syntax.ILogOptions;
import org.vcs.bazaar.client.commandline.syntax.IShelveOptions;
import org.vcs.bazaar.client.core.BazaarClientException;
import org.vcs.bazaar.client.core.BranchLocation;
import org.vcs.bazaar.client.testUtils.BazaarTest;
import org.vcs.bazaar.client.testUtils.Environment;
import org.vcs.bazaar.client.testUtils.ExpectedWorkingTree;
import org.vcs.bazaar.client.utils.BazaarUtilities;
import org.vcs.bazaar.client.utils.StringUtil;

/**
 * @author Guillermo Gonzalez
 *
 */
public class BazaarClientTest extends BazaarTest {

	/**
	 * Test method for
	 * {@link org.vcs.bazaar.client.commandline.CommandLineClient#add(java.io.File[], boolean)}.
	 * @throws Exception when something goes wrong
	 */
	@Test public final void testAdd() throws Exception {
		Environment testEnv;
		testEnv = new Environment("basicAdd", getTestConfig());
		client.setWorkDir(testEnv.getWorkingTreeLocation());

		File newDir = new File(testEnv.getWorkingTreeLocation(), "new_Dir");
		newDir.mkdirs();
		File newFile = new File(newDir, "new_File.txt");
		addContentToFile(newFile, "a new File");

		client.add(new File[] { newDir });
		testEnv.getExpectedWorkingTree().addItem("new_Dir", null);
		testEnv.getExpectedWorkingTree().setItemStatus("new_Dir", BazaarStatusKind.CREATED);
		testEnv.getExpectedWorkingTree().addItem("new_Dir/new_File.txt", "a new File");
		testEnv.getExpectedWorkingTree().addItem("new_Dir/a file with spaces in its name.txt", "a file with spaces");
		testEnv.getExpectedWorkingTree().setItemStatus("new_Dir/new_File.txt", BazaarStatusKind.CREATED);

		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
	}

	/**
	 * Test method for
	 * {@link org.vcs.bazaar.client.commandline.CommandLineClient#annotate(File, org.vcs.bazaar.client.commandline.commands.options.Option...)}.
	 * @throws Exception if something fails
	 */
	@Test public final void testAnnotate() throws Exception {
	    Environment testEnv = new Environment("basicAnnotate", getTestConfig());
	    client.setWorkDir(testEnv.getWorkingTreeLocation());

	    File file = new File(testEnv.getWorkingTreeLocation(), "A/file_in_A");
	    IBazaarAnnotation ann = client.annotate(file);

	    // this should be 1 line
	    String[] expectedContent = testEnv.getExpectedWorkingTree().getItemContent("A/file_in_A").split(System.getProperty("line.separator"));

	    Assert.assertEquals("Line count", expectedContent.length, ann.getNumberOfLines());
	    Assert.assertEquals("File path", file.getCanonicalPath(), ann.getFile().getCanonicalPath());
	    for (int i = 0; i < expectedContent.length; i++) {
	        Assert.assertEquals("Author", getBzrMail(), ann.getAuthor(i));
	        Assert.assertEquals("Code line", expectedContent[i], ann.getline(i));
	        // TODO: ann.getRevision(i).equals(obj)
	    }
	}

	/**
	 * Test method for
	 * {@link org.vcs.bazaar.client.commandline.CommandLineClient#branch(java.io.File, java.io.File)}.
	 */
	@Test public final void testBranch() throws Exception {
		Environment testEnv = new Environment("localBranch/original", getTestConfig());

		File newBranch = new File(testEnv.getWorkingTreeLocation().getParent(), "branch");
		client.setWorkDir(newBranch.getParentFile());
		client.branch(new BranchLocation(testEnv.getWorkingTreeLocation()), newBranch, null);
		assertWTEqual(testEnv.getWorkingTreeLocation(), newBranch);
	}

	/**
	 * Test method for
	 * {@link org.vcs.bazaar.client.commandline.CommandLineClient#cat(java.io.File, org.vcs.bazaar.client.BazaarRevision)}.
	 */
	@Test public final void testCat() throws Exception {
		Environment testEnv = new Environment("simpleCat", getTestConfig());
		client.setWorkDir(testEnv.getWorkingTreeLocation());

		File file = new File(testEnv.getWorkingTreeLocation(), "file_in_root.txt");
		String expectedContent = testEnv.getExpectedWorkingTree().getItemContent("file_in_root.txt");
		BufferedReader reader = new BufferedReader(new InputStreamReader(client.cat(file, null)));
		String line = null;
		final StringBuilder strBuilder = new StringBuilder();
		while ((line = reader.readLine()) != null) {
			strBuilder.append(line);
		}
		Assert.assertEquals(expectedContent.trim(), strBuilder.toString().trim());
	}

	@Test public final void testCheckout() throws Exception {
		Environment testEnv = new Environment("checkout/original", getTestConfig());

		File checkout = new File(testEnv.getWorkingTreeLocation().getParent(), "checkout");
		client.setWorkDir(checkout.getParentFile());

		BranchLocation branchLocation = new BranchLocation(testEnv.getWorkingTreeLocation().toURI());
		client.checkout(branchLocation, checkout);
		assertWTEqual(testEnv.getWorkingTreeLocation(), checkout);

		// now do a commit in the checkedout branch, and see if it's
		// commited to the parent branch
		String textToAdd = "this is added text"+System.getProperty("line.separator");
		File fileInRoot = new File(checkout, "file_in_root.txt");
		addContentToFile(fileInRoot, textToAdd);
		client.setWorkDir(checkout);
		client.commit(new File[] { fileInRoot }, "commit made in the checkedout branch to test client.checkout(lightweight=true)");
		client.setWorkDir(testEnv.getWorkingTreeLocation().getParentFile());
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
		testEnv.checkStatusesExpectedWorkingTree(checkout);
	}

	/**
	 * Test method for
	 * {@link org.vcs.bazaar.client.commandline.CommandLineClient#commit(java.io.File[], java.lang.String)}.
	 */
	@Test public final void testCommit() throws Exception {
		Environment testEnv = new Environment("basicCommit", getTestConfig());
		BazaarRevision expectedRevision = BazaarRevision.getRevision(2);
		File fileInRoot = setUpCommitTest(testEnv, expectedRevision);

		// commit the changes
		client.commit(new File[] { fileInRoot }, "file_in_root.txt modified by testCommit");

		// change the expectedWorkingTree to the expected status post commit
		testEnv.getExpectedWorkingTree().setItemRevision(fileInRoot.getName(), expectedRevision);
		Assert.assertEquals("current revision diffier from expected", expectedRevision, client.revno(fileInRoot));

		// Check if status is ok
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
	}

	@Test
	public final void testCommitEncoding() throws Exception {
		Environment testEnv = new Environment("MessageVariantsCommit", getTestConfig());
		BazaarRevision expectedRevision = BazaarRevision.getRevision(2);
		File fileInRoot = setUpCommitTest(testEnv, expectedRevision);
		String commitMessage = "Příšrně žlutý kůň úpěl ďábelské ódy (czech language)";
		client.commit(new File[] { fileInRoot }, commitMessage, ICommitOptions.UNCHANGED);
		// change the expectedWorkingTree to the expected status post commit
		testEnv.getExpectedWorkingTree().setItemRevision(fileInRoot.getName(), expectedRevision);
		Assert.assertEquals("current revision diffier from expected", expectedRevision, client.revno(fileInRoot));
		// Check if status is ok
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());

		List<IBazaarLogMessage> logs = client.log(fileInRoot, ILogOptions.LIMIT.with("1"));
		if(logs.size() == 1) {
			Assert.assertArrayEquals(commitMessage.toCharArray(), logs.get(0).getMessage().toCharArray());
			Assert.assertEquals(commitMessage, logs.get(0).getMessage());
		} else {
			Assert.fail("At least one log message is expected");
		}
	}

	@Test
	@Ignore(value="This test is failing due to a bad conversion of some control chars between xmloutput and bzr-java-lib")
	public final void testCommitMessageVariants() throws Exception {
		Environment testEnv = new Environment("MessageVariantsCommit", getTestConfig());
		BazaarRevision expectedRevision = BazaarRevision.getRevision(2);
		File fileInRoot = setUpCommitTest(testEnv, expectedRevision);
		String commitMessage = "file_in_root.txt \n \b \t \f \r \r\n \r \r \r \\ testCommit < ''| ";
		client.commit(new File[] { fileInRoot }, commitMessage, ICommitOptions.UNCHANGED);
		// change the expectedWorkingTree to the expected status post commit
		testEnv.getExpectedWorkingTree().setItemRevision(fileInRoot.getName(), expectedRevision);
		Assert.assertEquals("current revision diffier from expected", expectedRevision, client.revno(fileInRoot));
		// Check if status is ok
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());

		List<IBazaarLogMessage> logs = client.log(fileInRoot, ILogOptions.LIMIT.with("1"));
		if(logs.size() == 1) {
			char[] expected = commitMessage.toCharArray();
			char[] actual = logs.get(0).getMessage().toCharArray();
			Assert.assertEquals(expected.length, actual.length);
			for (int i = 0; i < expected.length; i++) {
				Assert.assertEquals("expected: "+expected[i]+"("+Integer.toHexString(expected[i])+") but was: "+actual[i]+"("+Integer.toHexString(actual[i])+") at char["+i+"]", expected[i], actual[i]);
			}
			Assert.assertEquals(commitMessage, logs.get(0).getMessage());
		} else {
			Assert.fail("At least one log message is expected");
		}
	}

	/**
	 * Helper to avoid copy & paste of the commit setup steps
	 * @param testEnv
	 * @param expectedRevision
	 * @return File added in the root
	 * @throws Exception
	 */
	private File setUpCommitTest(Environment testEnv, BazaarRevision expectedRevision) throws Exception {
		client.setWorkDir(testEnv.getWorkingTreeLocation());

		String textToAdd = "this is text added to file_in_root.txt"+System.getProperty("line.separator");

		// modify a existing file
		File fileInRoot = new File(testEnv.getWorkingTreeLocation(), "file_in_root.txt");
		addContentToFile(fileInRoot, textToAdd);

		// change the expectedWorkingTree to the expected status
		testEnv.getExpectedWorkingTree().setItemStatus(fileInRoot.getName(), BazaarStatusKind.MODIFIED);
		String currentContent = testEnv.getExpectedWorkingTree().getItemContent(fileInRoot.getName());
		testEnv.getExpectedWorkingTree().setItemContent(fileInRoot.getName(), currentContent + textToAdd);

		// Check if status is ok
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
		return fileInRoot;
	}

	/**
	 * Test method for
	 * {@link org.vcs.bazaar.client.commandline.CommandLineClient#diff(java.io.File, org.vcs.bazaar.client.BazaarRevision, org.vcs.bazaar.client.BazaarRevision)}.
	 */
	@Test public final void testDiff() throws Exception {
		Environment testEnv = new Environment("Diff", getTestConfig());
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		File file = new File(testEnv.getWorkingTreeLocation(), "A/file_in_A");
		String textToAdd = System.getProperty("line.separator")+"content added to test diff"+System.getProperty("line.separator");
		addContentToFile(file, textToAdd);
		String aDiff = client.diff(new File[]{file}, null);
		Assert.assertTrue(aDiff.contains("+content added to test diff"));
	}

	/**
	 * Test method for
	 * {@link org.vcs.bazaar.client.commandline.CommandLineClient#init(java.io.File)}.
	 */
	@Test public final void testInit() throws Exception {
		Environment testEnv = new Environment("simpleInit", getTestConfig());
		client.setWorkDir(testEnv.getWorkingTreeLocation().getParentFile());
		try {
			client.init(testEnv.getWorkingTreeLocation());
		} catch (BazaarClientException bzrCliEx) {
			Assert.assertNotNull(bzrCliEx.getMessage());
			Assert.assertTrue(bzrCliEx.getMessage().contains("Already a branch:"));
		}
	}

	/**
	 * Test method for
	 * {@link CommandLineClient#log(File, org.vcs.bazaar.client.commandline.commands.options.Option...)}.
	 */
	@Test public final void testLog() throws Exception {
	    Environment testEnv = new Environment("log", getTestConfig());
	    client.setWorkDir(testEnv.getWorkingTreeLocation());
	    List<IBazaarLogMessage> logMsgs = client.log(testEnv.getWorkingTreeLocation());
	    Assert.assertEquals(1, logMsgs.size());
	    for (IBazaarLogMessage message : logMsgs) {
	        Assert.assertEquals("default_branch", message.getBranchNick());
	        Assert.assertEquals(BazaarRevision.getRevision(1).getValue(), message.getRevision().getValue());
	        Assert.assertEquals(getUserName(), message.getCommiter());
	        Assert.assertEquals("initial import", message.getMessage().trim());
	    }
	}

	/**
	 * Test method for
	 * {@link org.vcs.bazaar.client.commandline.CommandLineClient#move(java.io.File, java.io.File)}.
	 */
	@Test public final void testMove() throws Exception {
		Environment testEnv = new Environment("simpleMove", getTestConfig());

		client.setWorkDir(testEnv.getWorkingTreeLocation());
		// try to move one file
		File orig = new File(testEnv.getWorkingTreeLocation(), "file_in_root.txt");
		File dest = new File(testEnv.getWorkingTreeLocation(), "A/");
		client.move(orig, dest);

		testEnv.getExpectedWorkingTree().addItem("A/file_in_root.txt", testEnv.getExpectedWorkingTree().getItemContent("file_in_root.txt"));
		testEnv.getExpectedWorkingTree().removeItem("file_in_root.txt");
		testEnv.getExpectedWorkingTree().setItemStatus("A/file_in_root.txt", BazaarStatusKind.RENAMED);

		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());

		// now try to move >1 files
		File[] filesToMove = new File[2];
		filesToMove[0] = new File(testEnv.getWorkingTreeLocation(), "A/B/file_in_B");
		filesToMove[1] = new File(testEnv.getWorkingTreeLocation(), "A/file_in_A");
		File destDir = testEnv.getWorkingTreeLocation(); // the branch
		// root
		client.move(filesToMove, destDir);

		testEnv.getExpectedWorkingTree().addItem("file_in_B", testEnv.getExpectedWorkingTree().getItemContent("A/B/file_in_B"));
		testEnv.getExpectedWorkingTree().addItem("file_in_A", testEnv.getExpectedWorkingTree().getItemContent("A/file_in_A"));
		testEnv.getExpectedWorkingTree().removeItem("A/B/file_in_B");
		testEnv.getExpectedWorkingTree().removeItem("A/file_in_A");
		testEnv.getExpectedWorkingTree().setItemStatus("file_in_B", BazaarStatusKind.RENAMED);
		testEnv.getExpectedWorkingTree().setItemStatus("file_in_A", BazaarStatusKind.RENAMED);

		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());

		// and the grand finale, try to move a directory
		File origDir = new File(testEnv.getWorkingTreeLocation(), "A/B/C");
		destDir = new File(testEnv.getWorkingTreeLocation(), "A/E/");
		client.move(origDir, destDir);

		// null content represents a directory
		testEnv.getExpectedWorkingTree().addItem("A/E/C", null);
		testEnv.getExpectedWorkingTree().setItemStatus("A/E/C", BazaarStatusKind.RENAMED);

		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
	}

	/**
	 * Test method for
	 * {@link org.vcs.bazaar.client.commandline.CommandLineClient#nick(java.io.File, java.lang.String)}.
	 */
	@Test public final void testNick() throws Exception {
		Environment testEnv = new Environment("simpleNick", getTestConfig());
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		String nick = client.nick(null);
		Assert.assertEquals(testEnv.getWorkingTreeLocation().getName().trim(), nick);
	}

	/**
	 * Test method for
	 * {@link org.vcs.bazaar.client.commandline.CommandLineClient#pull(java.net.URI, boolean)}.
	 */
	@Test public final void testPull() throws Exception {
		Environment testEnv = new Environment("Pull/parent", getTestConfig());

		File branch = new File(testEnv.getWorkingTreeLocation().getParent(), "branched");
		client.branch(new BranchLocation(testEnv.getWorkingTreeLocation()), branch, null);
		// first check if the parent branch and the checkout have the same
		// structure
		assertWTEqual(testEnv.getWorkingTreeLocation(), branch);

		client.setWorkDir(testEnv.getWorkingTreeLocation());
		// now do a commit in the checkedout branch, and see if it's
		// commited to the parent branch
		String textToAdd = "this is added text"+System.getProperty("line.separator");
		File fileInParent = new File(testEnv.getWorkingTreeLocation(), "file_in_root.txt");
		addContentToFile(fileInParent, textToAdd);
		client.add(new File[] { fileInParent });
		client.commit(new File[] { fileInParent }, "commit made in the parent branch to test client.pull");
		testEnv.getExpectedWorkingTree().addItem(fileInParent.getName(),
				testEnv.getExpectedWorkingTree().getItemContent(fileInParent.getName()) + textToAdd);

		client.setWorkDir(testEnv.getWorkingTreeLocation().getParentFile());
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
		testEnv.checkStatusesExpectedWorkingTree(branch);

		client.setWorkDir(branch);
		client.pull(testEnv.getWorkingTreeLocation().toURI());
		// the revno 2 should be pulled to branched
		Assert.assertEquals(client.revno(testEnv.getWorkingTreeLocation()), client.revno(branch));
	}

	/**
	 * Test method for
	 * {@link org.vcs.bazaar.client.commandline.CommandLineClient#push(URI, boolean, boolean, boolean, boolean)}.
	 */
	@Test public final void testPush() throws Exception {
		Environment testEnv = new Environment("push/remote", getTestConfig());

		File localBranch = new File(testEnv.getWorkingTreeLocation().getParent(), "local");
		client.branch(new BranchLocation(testEnv.getWorkingTreeLocation()), localBranch, null);
		assertWTEqual(testEnv.getWorkingTreeLocation(), localBranch);

		client.setWorkDir(localBranch);
		// now do a commit in the checkedout branch, and see if it's
		// commited to the parent branch
		String textToAdd = "this is added text"+System.getProperty("line.separator");
		File fileInParent = new File(localBranch, "file_in_root.txt");
		addContentToFile(fileInParent, textToAdd);
		client.add(new File[] { fileInParent });
		client.commit(new File[] { fileInParent }, "commit made in the local branch to test client.push");
		testEnv.getExpectedWorkingTree().addItem(fileInParent.getName(),
				testEnv.getExpectedWorkingTree().getItemContent(fileInParent.getName()) + textToAdd);

		client.setWorkDir(testEnv.getWorkingTreeLocation().getParentFile());
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
		testEnv.checkStatusesExpectedWorkingTree(localBranch);

		client.setWorkDir(localBranch);
		client.push(StringUtil.getAbsoluteURI(testEnv.getWorkingTreeLocation().toURI()));

		// the "remote" branch should be in revno 2
		Assert.assertEquals(client.revno(localBranch), client.revno(testEnv.getWorkingTreeLocation()));
	}

	/**
	 * Test method for
	 * {@link org.vcs.bazaar.client.commandline.CommandLineClient#remove(java.io.File[], boolean)}.
	 */
	@Test public final void testRemove() throws Exception {
		Environment testEnv = new Environment("singleFileRemove", getTestConfig());
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		// modify a existing file
		File fileInRoot = new File(testEnv.getWorkingTreeLocation(), "file_in_root.txt");
		Assert.assertTrue(fileInRoot.delete());
		testEnv.getExpectedWorkingTree().setItemStatus(fileInRoot.getName(), BazaarStatusKind.DELETED);
		// Check if status is ok after modifications
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
	}

	/**
	 * Test method for
	 * {@link org.vcs.bazaar.client.commandline.CommandLineClient#revert(java.io.File[])}.
	 */
	@Test public final void testRevert() throws Exception {
		Environment testEnv = new Environment("singleFileRevert", getTestConfig());
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		// modify a existing file
		File fileInRoot = new File(testEnv.getWorkingTreeLocation(), "file_in_root.txt");
		String textToAdd = "///added text\\\\\\"+System.getProperty("line.separator");
		addContentToFile(fileInRoot, textToAdd);
		// change the expectedWorkingTree to the expected status
		testEnv.getExpectedWorkingTree().setItemStatus(fileInRoot.getName(), BazaarStatusKind.MODIFIED);
		String previousContent = testEnv.getExpectedWorkingTree().getItemContent(fileInRoot.getName());
		testEnv.getExpectedWorkingTree().setItemContent(fileInRoot.getName(), previousContent + textToAdd);

		// Check if status is ok after modifications
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
		// commit the changes
		client.commit(new File[] { fileInRoot }, fileInRoot.getName() + " modified by testRevert");
		// change the expectedWorkingTree to the expected status post a
		// revert
		testEnv.getExpectedWorkingTree().setItemRevision(fileInRoot.getName(), BazaarRevision.getRevision(2));
		Assert.assertEquals("current revision diffier from expected", BazaarRevision.getRevision(2), client.revno(fileInRoot));
		// Check if status is ok
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());

		// now execute a 'bzr revert'
		client.revert(new File[] { fileInRoot });
		// setup expected status
		testEnv.getExpectedWorkingTree().setItemRevision(fileInRoot.getName(), BazaarRevision.getRevision(1));
		testEnv.getExpectedWorkingTree().setItemStatus(fileInRoot.getName(), BazaarStatusKind.UNCHANGED);
		testEnv.getExpectedWorkingTree().setItemContent(fileInRoot.getName(), previousContent);
		// check if all is ok
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
	}

	/**
	 * Test method for
	 * {@link org.vcs.bazaar.client.commandline.CommandLineClient#revno(java.io.File)}.
	 */
	@Test public final void testRevno() throws Exception {
		Environment testEnv = new Environment("Revno", getTestConfig());
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		BazaarRevision expectedRevision = BazaarRevision.getRevision(1);
		testEnv.getExpectedWorkingTree().setItemsRevision(expectedRevision);
		BazaarRevision result = client.revno(testEnv.getWorkingTreeLocation());
		Assert.assertEquals(Integer.valueOf(expectedRevision.getValue().trim()), Integer.valueOf(result.getValue().trim()));
	}

	/**
	 * This is a tricky thing to test, because by their nature
	 * revision identifiers will change every time you run the test.
	 * TODO : Think about ways of making sure it works better, like
	 * checking that the user id is in the revision-id
	 * @throws Exception
	 * @throws BazaarClientException
	 * @throws IOException
	 *
	 */
	@Test public final void testRevisionInfo() throws IOException, BazaarClientException, Exception {
		Environment testEnv = new Environment("RevisionInfo", getTestConfig());
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		BazaarRevision result = client.revisionInfo(testEnv.getWorkingTreeLocation(), null);
		assertNotNull(result);
		assertNotSame(result, BazaarRevision.INVALID);
	}

	/**
	 * Test method for
	 * {@link org.vcs.bazaar.client.commandline.CommandLineClient#status(java.io.File[])}.
	 */
	@Test public final void testStatus() throws Exception {
		Environment testEnv = new Environment("Status", getTestConfig());
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		File fileInRoot = new File(testEnv.getWorkingTreeLocation(), "file_in_root.txt");
		File fileInA = new File(testEnv.getWorkingTreeLocation(), "A/file_in_A");
		String newContent = "added content"+System.getProperty("line.separator");
		addContentToFile(fileInRoot, newContent);
		addContentToFile(fileInA, newContent);
		BazaarTreeStatus status = client.status(new File[] { fileInRoot, fileInA });
		Assert.assertEquals("Expected status info for 2 files but..", 2, status.getStatus().size());
		for (IBazaarStatus elem : status.getStatus()) {
			File absoluteFile = new File(elem.getBranchRoot(), elem.getFile().getPath());
			if (absoluteFile.getCanonicalPath().equals(fileInRoot.getCanonicalPath()) || absoluteFile.getCanonicalPath().equals(fileInA.getCanonicalPath())) {
				continue;
			} else {
				Assert.fail("recieved status for unexpected files!");
			}
		}
	}

	/**
	 * Test method for
	 * {@link org.vcs.bazaar.client.commandline.CommandLineClient#unCommit(java.io.File)}.
	 */
	@Test public final void testUnCommit() throws Exception {
		Environment testEnv = new Environment("singleFileRevert", getTestConfig());
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		// modify a existing file
		File fileInRoot = new File(testEnv.getWorkingTreeLocation(), "file_in_root.txt");
		String textToAdd = "///added text\\\\\\"+System.getProperty("line.separator");
		addContentToFile(fileInRoot, textToAdd);
		// change the expectedWorkingTree to the expected status
		testEnv.getExpectedWorkingTree().setItemStatus(fileInRoot.getName(), BazaarStatusKind.MODIFIED);
		String previousContent = testEnv.getExpectedWorkingTree().getItemContent(fileInRoot.getName());
		testEnv.getExpectedWorkingTree().setItemContent(fileInRoot.getName(), previousContent + textToAdd);

		// Check if status is ok after modifications
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
		// commit the changes
		client.commit(new File[] { fileInRoot }, fileInRoot.getName() + " modified by testCommit");
		// change the expectedWorkingTree to the expected status post a
		// revert
		testEnv.getExpectedWorkingTree().setItemRevision(fileInRoot.getName(), BazaarRevision.getRevision(2));
		Assert.assertEquals("current revision diffier from expected", BazaarRevision.getRevision(2), client.revno(fileInRoot));
		// Check if status is ok
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());

		// now execute a 'bzr uncommit'
		client.unCommit(testEnv.getWorkingTreeLocation());

			// status should be exactly the same previous the uncommit
			testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
	}

	/**
	 * Test method for
	 * {@link org.vcs.bazaar.client.commandline.CommandLineClient#ignore(java.lang.String)}.
	 */
	@Test public final void testIgnore() throws Exception {
		Environment testEnv = new Environment("Ignore", getTestConfig());
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		File newFile = new File(testEnv.getWorkingTreeLocation(), "file_to_ignore");
		String textToAdd = "///added text\\\\\\"+System.getProperty("line.separator");
		addContentToFile(newFile, textToAdd);
		testEnv.getExpectedWorkingTree().addItem(newFile.getName(), textToAdd);
		testEnv.getExpectedWorkingTree().setItemStatus(newFile.getName(), BazaarStatusKind.UNKNOWN);
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());

		client.ignore(testEnv.getWorkingTreeLocation(), newFile.getName());

		testEnv.getExpectedWorkingTree().removeItem(newFile.getName());
		testEnv.getExpectedWorkingTree().addItem(".bzrignore", newFile.getName() + "\n"); // ignore always uses *nix linefeeds
		testEnv.getExpectedWorkingTree().setItemStatus(".bzrignore", BazaarStatusKind.CREATED);

		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
	}

	/**
	 * Test method for
	 * {@link org.vcs.bazaar.client.commandline.CommandLineClient#ignored()}.
	 */
	@Test public final void testIgnored() throws Exception {
		Environment testEnv = new Environment("Ignored", getTestConfig());
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		File newFile = new File(testEnv.getWorkingTreeLocation(), "file_to_ignore");
		String textToAdd = "///added text\\\\\\"+System.getProperty("line.separator");
		addContentToFile(newFile, textToAdd);
		testEnv.getExpectedWorkingTree().addItem(newFile.getName(), textToAdd);
		testEnv.getExpectedWorkingTree().setItemStatus(newFile.getName(), BazaarStatusKind.UNKNOWN);
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());

		client.ignore(testEnv.getWorkingTreeLocation(), newFile.getName());

		testEnv.getExpectedWorkingTree().removeItem(newFile.getName());
		testEnv.getExpectedWorkingTree().addItem(".bzrignore", newFile.getName() + "\n"); // ignore always uses *nix line endings
		testEnv.getExpectedWorkingTree().setItemStatus(".bzrignore", BazaarStatusKind.CREATED);

		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());

		// until here, the same code as in testIgnored
		client.commit(new File[] { new File(testEnv.getWorkingTreeLocation(), ".bzrignore") }, "Added .bzrignore");
		testEnv.getExpectedWorkingTree().setItemStatus(".bzrignore", BazaarStatusKind.NONE);

		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());

		Map<String, String> result = client.ignored(testEnv.getWorkingTreeLocation());
		String unixPath = BazaarUtilities.unixFilePath(newFile);
		Assert.assertNotNull(result.get(unixPath));
		Assert.assertEquals(1, result.keySet().size());
		Assert.assertEquals(newFile.getName(), result.get(unixPath));

		// check that ignored also works on large filenames
		File a_dir = new File(testEnv.getWorkingTreeLocation(), "a_quite_long_named_directory");
		testEnv.getExpectedWorkingTree().addItem(a_dir.getName(), null);
		testEnv.getExpectedWorkingTree().setItemStatus(a_dir.getName(), BazaarStatusKind.CREATED);
		a_dir = new File(a_dir, "a_quite_long_named_directory");
		String a_dir_relativePath = BazaarUtilities.getRelativeTo(testEnv.getWorkingTreeLocation(), a_dir).getPath();
		testEnv.getExpectedWorkingTree().addItem(a_dir_relativePath, null);
		testEnv.getExpectedWorkingTree().setItemStatus(a_dir_relativePath, BazaarStatusKind.CREATED);
		a_dir.mkdirs();
		client.add(new File[]{a_dir});
		client.commit(new File[]{a_dir}, "add the dir");

		// create a file so we get a quite long relative path
		File largeFile = new File(a_dir, "file_to_ignore_with_a_long_name");
		largeFile.createNewFile();
		File largeRelFile = new File(a_dir_relativePath, largeFile.getName());
		String unixRelPath = BazaarUtilities.unixFilePath(largeRelFile);
		testEnv.getExpectedWorkingTree().addItem(unixRelPath, "");
		testEnv.getExpectedWorkingTree().setItemStatus(unixRelPath, BazaarStatusKind.UNKNOWN);

		// check the expected WT
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());

		client.ignore(testEnv.getWorkingTreeLocation(), largeRelFile.getPath());
		result = client.ignored(testEnv.getWorkingTreeLocation());
		Assert.assertNotNull(result.get(BazaarUtilities.unixFilePath(largeFile)));
		Assert.assertEquals(2, result.keySet().size());
		Assert.assertEquals(unixRelPath, result.get(BazaarUtilities.unixFilePath(largeFile)));

	}

	@Test public final void testUnknowns() throws Exception {
		Environment testEnv = new Environment("unknowns", getTestConfig());
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		File fileInRoot = new File(testEnv.getWorkingTreeLocation(), "file_in_root_new.txt");
		File fileInA = new File(testEnv.getWorkingTreeLocation(), "A/file_in_A_new");
		fileInRoot.createNewFile();
		fileInA.createNewFile();
		String[] unknowns = client.unknowns(testEnv.getWorkingTreeLocation());
		Assert.assertEquals("Expected status info for 2 files but..", 2, unknowns.length);
		BazaarTreeStatus status = client.status(null);
		Assert.assertEquals("Expected status info for 2 files but..", 2, status.getStatus().size());
		for (IBazaarStatus elem : status.getStatus()) {
			if (!elem.contains(BazaarStatusKind.UNKNOWN)) {
				Assert.fail("file: " + elem.getAbsolutePath() + " isn't unknown");
			}
		}
	}

	@Test public final void testInfo() throws Exception {
		Environment testEnv = new Environment("info", getTestConfig());
		IBazaarInfo info = client.info(testEnv.getWorkingTreeLocation(), IInfoOptions.VERBOSE);
		assertNotNull(info);
		assertNotNull(info.getBranchFormat());
		assertNotNull(info.getBranchHistory());
		assertNotNull(info.getControlFormat());
		assertNotNull(info.getLayout());
		assertNotNull(info.getRepositoryFormat());
		assertNotNull(info.getWorkingTreeFormat());
		assertNotNull(info.getFormats());
		assertNotNull(info.getLocations());
		assertNotNull(info.getRelatedBranches());
		assertNotNull(info.getRepositoryStats());
		assertNotNull(info.getWorkingTreeStats());
	}

	@Test
	public final void testBind() throws Exception {
		Environment testEnv = new Environment("bind", getTestConfig());
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		File branch = new File(testEnv.getWorkingTreeLocation().getParent(), "branch_bind");
		BranchLocation source = new BranchLocation(testEnv.getWorkingTreeLocation());
		client.branch(source, branch, null);
		assertWTEqual(testEnv.getWorkingTreeLocation(), branch);
		IBazaarInfo info = client.info(branch);
		assertFalse(testEnv.getWorkingTreeLocation().getAbsolutePath().equals(info.getLocations().getCheckoutOfBranch()));
		client.setWorkDir(branch);
		client.bind(source);
		info = client.info(branch);
		assertEquals(BazaarUtilities.unixFilePath(testEnv.getWorkingTreeLocation().getAbsoluteFile()), info.getLocations().getCheckoutOfBranch());
	}

	@Test
	public final void testUnBind() throws Exception {
		Environment testEnv = new Environment("bind", getTestConfig());
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		File branch = new File(testEnv.getWorkingTreeLocation().getParent(), "branch_unbind");
		BranchLocation source = new BranchLocation(testEnv.getWorkingTreeLocation());
		client.checkout(source, branch);
		assertWTEqual(testEnv.getWorkingTreeLocation(), branch);
		client.setWorkDir(branch);
		IBazaarInfo info = client.info(branch);
		assertEquals(BazaarUtilities.unixFilePath(testEnv.getWorkingTreeLocation().getAbsoluteFile()), info.getLocations().getCheckoutOfBranch());
		client.unBind();
		info = client.info(branch);
		assertFalse(testEnv.getWorkingTreeLocation().getCanonicalPath().equals(info.getLocations().getCheckoutOfBranch()));
	}

	@Test
	public final void testSwitchBranch() throws Exception {
		Environment testEnv = new Environment("switch", getTestConfig());
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		File branch1 = new File(testEnv.getWorkingTreeLocation().getParent(), "branch1");
		File branch2 = new File(testEnv.getWorkingTreeLocation().getParent(), "branch2");
		BranchLocation source = new BranchLocation(testEnv.getWorkingTreeLocation());
		client.checkout(source, branch1);
		client.branch(source, branch2, null);
		client.setWorkDir(branch2);
		File fileToCommit = setUpCommitTest(testEnv, BazaarRevision.getRevision(2));
		client.commit(new File[]{fileToCommit}, "test commit");
		client.setWorkDir(branch1);
		IBazaarInfo info = client.info(new BranchLocation(branch1));
		assertEquals(BazaarUtilities.unixFilePath(testEnv.getWorkingTreeLocation().getAbsoluteFile()), info.getLocations().getCheckoutOfBranch());
		client.switchBranch(new BranchLocation(branch2));
		IBazaarInfo infoPostSwitch = client.info(new BranchLocation(branch1));
		assertEquals(BazaarUtilities.unixFilePath(branch2.getAbsoluteFile()), infoPostSwitch.getLocations().getCheckoutOfBranch());
		BazaarRevision expectedRevno = client.revno(new BranchLocation(branch2));
		BazaarRevision actualRevno = client.revno(new BranchLocation(branch1));
		assertEquals(expectedRevno, actualRevno);
	}

	@Test public final void testfindMergeBase() throws Exception {
		Environment testEnv = new Environment("findMergeBase/original", getTestConfig());

		File branch = new File(testEnv.getWorkingTreeLocation().getParent(), "branch");
		client.branch(new BranchLocation(testEnv.getWorkingTreeLocation()), branch, null);
		assertWTEqual(testEnv.getWorkingTreeLocation(), branch);

		BazaarRevision revision = client.findMergeBase(new BranchLocation(branch), new BranchLocation(testEnv.getWorkingTreeLocation()));
		assertNotNull(revision);
		List<IBazaarLogMessage> logs = client.log(new BranchLocation(testEnv.getWorkingTreeLocation()), ILogOptions.REVISION.with("1"), ILogOptions.SHOW_IDS);
		assertEquals(revision.getValue(), logs.get(0).getRevisionId());
	}

	@Test public final void testLs() throws Exception {
		Environment testEnv = new Environment("ls", getTestConfig());
		File wtPath = testEnv.getWorkingTreeLocation();
		client.setWorkDir(wtPath);
		IBazaarItemInfo[] items = client.ls(wtPath, null);
		ExpectedWorkingTree tree = testEnv.getExpectedWorkingTree();
		for (IBazaarItemInfo item : items) {
			File file = BazaarUtilities.getRelativeTo(wtPath, new File(item.getPath()));
			String path = unixFilePath(file);
			assertNotNull(path, tree.getItem(path));
			assertFalse("".equals(tree.getItem(path).getPath().trim()));
		}
	}

	@Test
	public final void testMissing() throws Exception {
		Environment testEnv = new Environment("missing", getTestConfig());

		File branch = new File(testEnv.getWorkingTreeLocation().getParent(), "missing_branched");
		client.branch(new BranchLocation(testEnv.getWorkingTreeLocation()), branch, null);
		assertWTEqual(testEnv.getWorkingTreeLocation(), branch);
		
		// sanity check, run missing with up to date branches
		client.setWorkDir(branch);
		Map<String, List<IBazaarLogMessage>> missing = client.missing(branch, new BranchLocation(StringUtil.getAbsoluteURI(testEnv.getWorkingTreeLocation())));
		assertNotNull(missing);
		assertFalse(missing.containsKey("MINE"));
		assertFalse(missing.containsKey("OTHER"));
		assertEquals(0, missing.size());

		String textToAdd = "this is added text"+System.getProperty("line.separator");
		File fileInParent = new File(testEnv.getWorkingTreeLocation(), "file_in_root.txt");
		addContentToFile(fileInParent, textToAdd);
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		client.add(new File[] { fileInParent });
		client.commit(new File[] { fileInParent }, "commit made in the parent branch to test missing");

		testEnv.getExpectedWorkingTree().addItem(fileInParent.getName(),
				testEnv.getExpectedWorkingTree().getItemContent(fileInParent.getName()) + textToAdd);

		client.setWorkDir(testEnv.getWorkingTreeLocation().getParentFile());
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
		testEnv.checkStatusesExpectedWorkingTree(branch);

		client.setWorkDir(branch);
		missing = client.missing(branch, new BranchLocation(StringUtil.getAbsoluteURI(testEnv.getWorkingTreeLocation())));
		assertNotNull(missing);
		assertNotNull(missing.get("OTHER"));
		assertEquals(1, missing.get("OTHER").size());

		client.setWorkDir(testEnv.getWorkingTreeLocation());
		missing = client.missing(branch, new BranchLocation(StringUtil.getAbsoluteURI(branch)));
		assertNotNull(missing);
		assertNotNull(missing.get("MINE"));
		assertEquals(1, missing.get("MINE").size());

		// add a revision to the other branch to test mine and other
		File fileInBranch = new File(branch, "file_in_root.txt");
		addContentToFile(fileInBranch, textToAdd);
		client.setWorkDir(branch);
		client.add(new File[] { fileInBranch });
		client.commit(new File[] { fileInBranch }, "commit made in the branch to test missing");

		client.setWorkDir(branch.getParentFile());
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
		testEnv.checkStatusesExpectedWorkingTree(branch);

		client.setWorkDir(branch);
		missing = client.missing(branch, new BranchLocation(StringUtil.getAbsoluteURI(testEnv.getWorkingTreeLocation())));
		assertNotNull(missing);
		assertNotNull(missing.get("MINE"));
		assertNotNull(missing.get("OTHER"));
		assertEquals(2, missing.size());
		assertEquals(1, missing.get("MINE").size());
		assertEquals(1, missing.get("OTHER").size());
	}

	@Test @Ignore public final void testUpdate() throws Exception {
		Assert.fail("not Implemented");
	}

	@Test
	public final void testMerge() throws Exception {
		Environment testEnv = new Environment("merge", getTestConfig());

		File branch = new File(testEnv.getWorkingTreeLocation().getParent(), "merge_branched");
		client.branch(new BranchLocation(testEnv.getWorkingTreeLocation()), branch, null);
		assertWTEqual(testEnv.getWorkingTreeLocation(), branch);

		String textToAdd = "this is added text"+System.getProperty("line.separator");
		File fileInParent = new File(testEnv.getWorkingTreeLocation(), "file_in_root.txt");
		addContentToFile(fileInParent, textToAdd);
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		client.add(new File[] { fileInParent });
		client.commit(new File[] { fileInParent }, "commit made in the parent branch to test a merge");

		testEnv.getExpectedWorkingTree().addItem(fileInParent.getName(),
				testEnv.getExpectedWorkingTree().getItemContent(fileInParent.getName()) + textToAdd);

		client.setWorkDir(testEnv.getWorkingTreeLocation().getParentFile());
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
		testEnv.checkStatusesExpectedWorkingTree(branch);

		client.setWorkDir(branch);
		client.merge(new BranchLocation(StringUtil.getAbsoluteURI(testEnv.getWorkingTreeLocation())));
		BazaarTreeStatus treeStatus = client.status(new File[]{branch});
		IBazaarStatus[] status = treeStatus.getStatusAsArray();
		Assert.assertEquals(1, status.length);
		client.commit(new File[]{branch}, "merge with parent");
		// both branches should be at revno 2
		Assert.assertEquals("2", client.revno(branch).getValue());
		Assert.assertEquals(client.revno(testEnv.getWorkingTreeLocation()), client.revno(branch));
	}

	@Test
	public final void testResolve() throws Exception {
		Environment testEnv = new Environment("resolve", getTestConfig());

		File branch = new File(testEnv.getWorkingTreeLocation().getParent(), "to_resolve");
		client.branch(new BranchLocation(testEnv.getWorkingTreeLocation()), branch, null);
		assertWTEqual(testEnv.getWorkingTreeLocation(), branch);

		String textToAdd = "this is added text"+System.getProperty("line.separator");
		File fileInParent = new File(testEnv.getWorkingTreeLocation(), "file_in_root.txt");
		addContentToFile(fileInParent, textToAdd);
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		client.add(new File[] { fileInParent });
		client.commit(new File[] { fileInParent }, "commit made in the parent branch to test a merge");

		testEnv.getExpectedWorkingTree().addItem(fileInParent.getName(),
				testEnv.getExpectedWorkingTree().getItemContent(fileInParent.getName()) + textToAdd);

		client.setWorkDir(testEnv.getWorkingTreeLocation().getParentFile());
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
		testEnv.checkStatusesExpectedWorkingTree(branch);

		client.setWorkDir(branch);
		File fileInBranch = new File(branch, "file_in_root.txt");
		addContentToFile(fileInBranch, textToAdd);
		addContentToFile(fileInBranch, textToAdd);
		client.commit(new File[] { fileInBranch }, "commit made in the branch to produce a conflict");

		try {
			client.merge(new BranchLocation(StringUtil.getAbsoluteURI(testEnv.getWorkingTreeLocation())));
		} catch(BazaarClientException e) {
			String msg = e.getMessage();
			Assert.assertTrue("content don't match", msg.contains("file_in_root.txt") && msg.contains("conflict"));
		}
		BazaarTreeStatus treeStatus = client.status(new File[]{branch});
		Assert.assertEquals(1, treeStatus.getStatus().size());
		IBazaarStatus[] status = treeStatus.getStatusAsArray();
		Assert.assertTrue("status[0] \""+ status[0] +"\" don't have any conflicts", status[0].contains(BazaarStatusKind.HAS_CONFLICTS));
		List<File> files = new ArrayList<File>(1);
		files.add(status[0].getFile());
		client.resolve(files);
		treeStatus = client.status(new File[]{branch});
		status = treeStatus.getStatus().toArray(new IBazaarStatus[0]);
		Assert.assertEquals(1, status.length);
		Assert.assertFalse("status[0] \""+ status[0] +"\" have a conflict", status[0].contains(BazaarStatusKind.HAS_CONFLICTS));
		client.commit(new File[]{branch}, "merge with conflict resolution");
		Assert.assertEquals("3", client.revno(branch).getValue());
	}

	@Test @Ignore public final void testSend() throws Exception {
		Assert.fail("not Implemented");
	}

	@Test public final void testVersionInfo() throws Exception {
		Environment testEnv = new Environment("version-info", getTestConfig());
		BazaarVersionInfo info = client.versionInfo(new BranchLocation(testEnv.getWorkingTreeLocation()));
		assertNotNull(info);
		assertNotNull(info.getBranchNick());
		assertNotNull(info.getBuildDate());
		assertNotNull(info.getDate());
		assertNotNull(info.getRevid());
		assertNotNull(info.getRevno());
	}

	@Test
	public final void testPlugins() throws IOException, BazaarClientException, Exception {
        Set<IPlugin> plugins = client.plugins();
	    Assert.assertTrue(plugins.size() >= 1);
	}

	@Test public final void testProgress() throws Exception {
		Assume.assumeTrue(CommandLineClientFactory.checkXmlOutputVersion(new int[]{0, 8, 9}));
		Environment testEnv = new Environment("progressBranch/origin", getTestConfig());
		File branch = new File(testEnv.getWorkingTreeLocation().getParent(), "branch");
		client.setWorkDir(branch.getParentFile());
		final AtomicReference<String> ref = new AtomicReference<String>();
		IBazaarProgressListener listener = new IBazaarProgressListener() {
			public void logProgress(String message) {
				if (message != null && !message.isEmpty()) {
					ref.set(message);
				}
			}
		};
		client.branch(new BranchLocation("lp:bzr-java-lib"), branch, null, listener);
		assertNotNull(ref.get());
		assertFalse(ref.get().isEmpty());
	}

	protected boolean isPluginInstalled(String pluginName) throws BazaarClientException {
		Set<IPlugin> plugins = client.plugins();
		for (IPlugin plugin : plugins) {
			if (pluginName.equals(plugin.getName())) {
				return true;
			}
		}
		return false;
	}

	@Test
	public final void testShelve() throws Exception {
		Assume.assumeTrue(CommandLineClientFactory.checkXmlOutputVersion(new int[]{0, 8, 9}));
		Environment testEnv = new Environment("shelve", getTestConfig());
		String textToAdd = "this is added text"+System.getProperty("line.separator");
		File fileToShelve = new File(testEnv.getWorkingTreeLocation(), "file_in_root.txt");
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		String content = testEnv.getExpectedWorkingTree().getItemContent(fileToShelve.getName());
		addContentToFile(fileToShelve, textToAdd);
		testEnv.getExpectedWorkingTree().setItemStatus(fileToShelve.getName(), BazaarStatusKind.MODIFIED);
		testEnv.getExpectedWorkingTree().setItemContent(fileToShelve.getName(), content + textToAdd);
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
		client.shelve(new File[]{fileToShelve}, IShelveOptions.ALL);
		testEnv.getExpectedWorkingTree().setItemStatus(fileToShelve.getName(), BazaarStatusKind.UNCHANGED);
		testEnv.getExpectedWorkingTree().setItemContent(fileToShelve.getName(), content);
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
		Assert.assertEquals(1, client.shelveList().size());
		client.unShelve(null);
		testEnv.getExpectedWorkingTree().setItemStatus(fileToShelve.getName(), BazaarStatusKind.MODIFIED);
		testEnv.getExpectedWorkingTree().setItemContent(fileToShelve.getName(), content + textToAdd);
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
		Assert.assertEquals(0, client.shelveList().size());
	}

	@Test
	public final void testRebase() throws Exception {

		Assume.assumeTrue(isPluginInstalled("rewrite"));
		Environment testEnv = new Environment("rebase", getTestConfig());
		File branch = new File(testEnv.getWorkingTreeLocation().getParent(), "rebase_branch");
		client.branch(new BranchLocation(testEnv.getWorkingTreeLocation()), branch, null);
		assertWTEqual(testEnv.getWorkingTreeLocation(), branch);

		String textToAdd = "this is added text"+System.getProperty("line.separator");
		File fileInParent = new File(testEnv.getWorkingTreeLocation(), "file_in_root.txt");
		addContentToFile(fileInParent, textToAdd);
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		client.add(new File[] { fileInParent });
		client.commit(new File[] { fileInParent }, "commit made in the parent branch to test a rebase");
		Assert.assertEquals("2", client.revno(testEnv.getWorkingTreeLocation()).getValue());
		testEnv.getExpectedWorkingTree().addItem(fileInParent.getName(),
				testEnv.getExpectedWorkingTree().getItemContent(fileInParent.getName()) + textToAdd);

		File fileInBranch = new File(branch, "file_in_branch.txt");
		addContentToFile(fileInBranch, textToAdd);
		client.setWorkDir(branch);
		client.add(new File[] { fileInBranch });
		client.commit(new File[] { fileInBranch }, "commit made in the child branch to test a rebase");

		client.setWorkDir(testEnv.getWorkingTreeLocation().getParentFile());
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
		testEnv.checkStatusesExpectedWorkingTree(branch);

		client.setWorkDir(testEnv.getWorkingTreeLocation());
		client.rebase(new BranchLocation(StringUtil.getAbsoluteURI(branch)));
		Assert.assertEquals("2", client.revno(branch).getValue());
		Assert.assertEquals("3", client.revno(testEnv.getWorkingTreeLocation()).getValue());
	}

	@Test
	public final void testRebaseContinue() throws Exception {
		Assume.assumeTrue(isPluginInstalled("rewrite"));
		Environment testEnv = new Environment("rebase", getTestConfig());
		File branch = new File(testEnv.getWorkingTreeLocation().getParent(), "rebase_continue_branch");
		client.branch(new BranchLocation(testEnv.getWorkingTreeLocation()), branch, null);
		assertWTEqual(testEnv.getWorkingTreeLocation(), branch);

		String textToAdd = "this is added text"+System.getProperty("line.separator");
		File fileInParent = new File(testEnv.getWorkingTreeLocation(), "file_in_root.txt");
		addContentToFile(fileInParent, textToAdd);
		client.setWorkDir(testEnv.getWorkingTreeLocation().getParentFile());
		client.add(new File[] { fileInParent });
		client.commit(new File[] { fileInParent }, "commit made in the parent branch to test a rebase");
		Assert.assertEquals("2", client.revno(testEnv.getWorkingTreeLocation()).getValue());
		testEnv.getExpectedWorkingTree().addItem(fileInParent.getName(),
				testEnv.getExpectedWorkingTree().getItemContent(fileInParent.getName()) + textToAdd);

		File fileInBranch = new File(branch, "file_in_root.txt");
		textToAdd = "another text to add"+System.getProperty("line.separator");
		addContentToFile(fileInBranch, textToAdd);
		client.setWorkDir(branch);
		client.add(new File[] { fileInBranch });
		client.commit(new File[] { fileInBranch }, "commit made in the child branch to test a rebase");

		client.setWorkDir(testEnv.getWorkingTreeLocation().getParentFile());
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
		testEnv.checkStatusesExpectedWorkingTree(branch);

		client.setWorkDir(testEnv.getWorkingTreeLocation());
		Assert.assertFalse(client.rebaseToDo());
		Assert.assertEquals("2", client.revno(branch).getValue());
		Assert.assertEquals("2", client.revno(testEnv.getWorkingTreeLocation()).getValue());
		try {
			client.rebase(new BranchLocation(StringUtil.getAbsoluteURI(branch)));
			Assert.fail("A conflict expected while running rebase");
		} catch (Exception e) {
		}
		Assert.assertEquals("2", client.revno(branch).getValue());
		Assert.assertEquals("2", client.revno(testEnv.getWorkingTreeLocation()).getValue());
		Assert.assertTrue(client.rebaseToDo());
		List<File> files = new ArrayList<File>();
		files.add(fileInParent);
		client.resolve(files);
		client.rebaseContinue();
		Assert.assertEquals("2", client.revno(branch).getValue());
		Assert.assertEquals("3", client.revno(testEnv.getWorkingTreeLocation()).getValue());
		Assert.assertFalse(client.rebaseToDo());
	}

	@Test
	public final void testRebaseAbort() throws Exception {
		Assume.assumeTrue(isPluginInstalled("rewrite"));
		Environment testEnv = new Environment("rebase", getTestConfig());
		File branch = new File(testEnv.getWorkingTreeLocation().getParent(), "rebase_abort_branch");
		client.branch(new BranchLocation(testEnv.getWorkingTreeLocation()), branch, null);
		assertWTEqual(testEnv.getWorkingTreeLocation(), branch);

		String textToAdd = "this is added text"+System.getProperty("line.separator");
		File fileInParent = new File(testEnv.getWorkingTreeLocation(), "file_in_root.txt");
		addContentToFile(fileInParent, textToAdd);
		client.setWorkDir(testEnv.getWorkingTreeLocation().getParentFile());
		client.add(new File[] { fileInParent });
		client.commit(new File[] { fileInParent }, "commit made in the parent branch to test a rebase");
		Assert.assertEquals("2", client.revno(testEnv.getWorkingTreeLocation()).getValue());
		testEnv.getExpectedWorkingTree().addItem(fileInParent.getName(),
				testEnv.getExpectedWorkingTree().getItemContent(fileInParent.getName()) + textToAdd);

		File fileInBranch = new File(branch, "file_in_branch.txt");
		textToAdd = "new file in branch"+System.getProperty("line.separator");
		addContentToFile(fileInBranch, textToAdd);
		client.setWorkDir(branch);
		client.add(new File[] { fileInBranch });
		client.commit(new File[] { fileInBranch }, "commit made in the child branch to test a rebase");

		fileInBranch = new File(branch, "file_in_root.txt");
		textToAdd = "another text to add"+System.getProperty("line.separator");
		addContentToFile(fileInBranch, textToAdd);
		client.setWorkDir(branch);
		client.add(new File[] { fileInBranch });
		client.commit(new File[] { fileInBranch }, "2nd commit made in the child branch to test a rebase");

		client.setWorkDir(testEnv.getWorkingTreeLocation().getParentFile());
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
		testEnv.checkStatusesExpectedWorkingTree(branch);

		client.setWorkDir(testEnv.getWorkingTreeLocation());
		Assert.assertFalse(client.rebaseToDo());
		Assert.assertEquals("3", client.revno(branch).getValue());
		Assert.assertEquals("2", client.revno(testEnv.getWorkingTreeLocation()).getValue());
		try {
			client.rebase(new BranchLocation(StringUtil.getAbsoluteURI(branch)));
			Assert.fail("A conflict expected while running rebase");
		} catch (Exception e) {
		}
		Assert.assertEquals("3", client.revno(branch).getValue());
		Assert.assertEquals("3", client.revno(testEnv.getWorkingTreeLocation()).getValue());
		Assert.assertTrue(client.rebaseToDo());
		List<File> files = new ArrayList<File>();
		files.add(fileInParent);
		client.rebaseAbort();
		Assert.assertEquals("3", client.revno(branch).getValue());
		Assert.assertEquals("2", client.revno(testEnv.getWorkingTreeLocation()).getValue());
		Assert.assertFalse(client.rebaseToDo());
	}

	@Test
	public final void testConflicts() throws Exception {
		Assume.assumeTrue(CommandLineClientFactory.checkXmlOutputVersion(new int[]{0, 8, 9}));
		Environment testEnv = new Environment("resolve", getTestConfig());

		File branch = new File(testEnv.getWorkingTreeLocation().getParent(), "conflicts");
		client.branch(new BranchLocation(testEnv.getWorkingTreeLocation()), branch, null);
		assertWTEqual(testEnv.getWorkingTreeLocation(), branch);

		String textToAdd = "this is added text"+System.getProperty("line.separator");
		File fileInParent = new File(testEnv.getWorkingTreeLocation(), "file_in_root.txt");
		addContentToFile(fileInParent, textToAdd);
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		client.add(new File[] { fileInParent });
		client.commit(new File[] { fileInParent }, "commit made in the parent branch to test a merge");

		testEnv.getExpectedWorkingTree().addItem(fileInParent.getName(),
				testEnv.getExpectedWorkingTree().getItemContent(fileInParent.getName()) + textToAdd);

		client.setWorkDir(testEnv.getWorkingTreeLocation().getParentFile());
		testEnv.checkStatusesExpectedWorkingTree(testEnv.getWorkingTreeLocation());
		testEnv.checkStatusesExpectedWorkingTree(branch);

		client.setWorkDir(branch);
		File fileInBranch = new File(branch, "file_in_root.txt");
		addContentToFile(fileInBranch, textToAdd);
		addContentToFile(fileInBranch, textToAdd);
		client.commit(new File[] { fileInBranch }, "commit made in the branch to produce a conflict");

		try {
			client.merge(new BranchLocation(StringUtil.getAbsoluteURI(testEnv.getWorkingTreeLocation())));
		} catch(BazaarClientException e) {
			String msg = e.getMessage();
			Assert.assertTrue("content don't match", msg.contains("file_in_root.txt") && msg.contains("conflict"));
		}
		BazaarTreeStatus treeStatus = client.status(new File[]{branch});
		Assert.assertEquals(1, treeStatus.getStatus().size());
		IBazaarStatus[] status = treeStatus.getStatusAsArray();
		Assert.assertTrue("status[0] \""+ status[0] +"\" don't have any conflicts", status[0].contains(BazaarStatusKind.HAS_CONFLICTS));
		List<IBazaarConflict> conflicts = client.conflicts();
		Assert.assertEquals(1, conflicts.size());
		List<File> files = new ArrayList<File>(1);
		files.add(status[0].getFile());
		client.resolve(files);
		treeStatus = client.status(new File[]{branch});
		status = treeStatus.getStatus().toArray(new IBazaarStatus[0]);
		Assert.assertEquals(1, status.length);
		Assert.assertFalse("status[0] \""+ status[0] +"\" have a conflict", status[0].contains(BazaarStatusKind.HAS_CONFLICTS));
		conflicts = client.conflicts();
		Assert.assertEquals(0, conflicts.size());
	}
}
