/**
 *
 */
package org.vcs.bazaar.client.utils;

import static org.junit.Assert.assertEquals;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.net.URI;

import org.junit.Test;

/**
 * @author Guillermo Gonzalez <guillo.gonzo AT gmail DOT com>
 *
 */
public class StringUtilTest {

	public static final String[] REMOTE_SCHEMES = new String[] { "http", "sftp", "ftp", "rsync", "https", "bzr+ssh",
		"bzr+http", "bzr+https", "https+urllib"};
	
	/**
	 * Test method for {@link org.vcs.bazaar.client.utils.StringUtil#getAbsoluteURI(java.io.File)}.
	 */
	@Test
	public final void testGetAbsoluteURIFile() {
		String path = "/Users/username/test_dir";
		if(BazaarUtilities.isWindows()) {
			path = "/c:/Users/username/test_dir";
		}
		File file = new File(path);
		URI expected = URI.create("file://" + path);
		URI actual = StringUtil.getAbsoluteURI(file);
		assertEquals(expected.toString(), actual.toString());
		file = new File(URI.create("file:" + path));
		actual = StringUtil.getAbsoluteURI(file);
		assertEquals(expected.toString(), actual.toString());
	}

	/**
	 * Test method for {@link org.vcs.bazaar.client.utils.StringUtil#getAbsoluteURI(java.net.URI)}.
	 */
	@Test
	public final void testGetAbsoluteURIURI() {
		URI expected = URI.create("file:///Users/username/test_dir");
		URI actual = StringUtil.getAbsoluteURI(URI.create("file:///Users/username/test_dir"));
		assertEquals(expected.toString(), actual.toString());
		actual = StringUtil.getAbsoluteURI(URI.create("file://Users/username/test_dir"));
		assertEquals(expected.toString(), actual.toString());
		actual = StringUtil.getAbsoluteURI(URI.create("file:/Users/username/test_dir"));
		assertEquals(expected.toString(), actual.toString());
	}

	@Test
	public final void testGetAbsoluteURISchemas() {
		String[] schemes = REMOTE_SCHEMES;
		for (String scheme : schemes) {
			URI expected = URI.create(scheme+"://Users/username/test_dir");
			URI actual = StringUtil.getAbsoluteURI(URI.create(scheme+"://Users/username/test_dir"));
			assertEquals(expected.toString(), actual.toString());
		}
		URI expected = URI.create("file:///Users/username/test_dir");
		URI actual = StringUtil.getAbsoluteURI(URI.create("file:///Users/username/test_dir"));
		assertEquals(expected.toString(), actual.toString());
	}

	@Test
	public final void testUrlEncodeWin() throws UnsupportedEncodingException {
		String raw = "file:///c:/Maçã/does/not/exist";
		String url = "file:///c:/Ma%C3%A7%C3%A3/does/not/exist";
		assertEquals(url, StringUtil.urlEncode(raw));

		raw = "file:///c:/non/ascii/terminated/maçã";
		url = "file:///c:/non/ascii/terminated/ma%C3%A7%C3%A3";
		assertEquals(url, StringUtil.urlEncode(raw));
	}

	@Test
	public final void testUrlEncodeUnix() throws UnsupportedEncodingException {
		String raw = "file:///home/Maçã/does/not/exist";
		String url = "file:///home/Ma%C3%A7%C3%A3/does/not/exist";
		assertEquals(url, StringUtil.urlEncode(raw));
		raw = "/home/Maçã/does/not/exist";
		url = "/home/Ma%C3%A7%C3%A3/does/not/exist";
		assertEquals(url, StringUtil.urlEncode(raw));
		raw = "lp:~Maçã/does/not/exist";
		url = "lp:~Ma%C3%A7%C3%A3/does/not/exist";
		assertEquals(url, StringUtil.urlEncode(raw));
		for (String scheme : REMOTE_SCHEMES) {
			raw = scheme+"://Maçã/does/not/exist";
			url = scheme+"://Ma%C3%A7%C3%A3/does/not/exist";
			assertEquals(url, StringUtil.urlEncode(raw));
		}
	}

	@Test
	public final void testUrlEncodeRemote() throws UnsupportedEncodingException {
		for (String scheme : REMOTE_SCHEMES) {
			String raw = scheme+"://Maçã/does/not/exist";
			String url = scheme+"://Ma%C3%A7%C3%A3/does/not/exist";
			assertEquals(url, StringUtil.urlEncode(raw));
		}
	}

}
