/**
 * LICENCSE + COPYRIGHT
 */
package org.vcs.bazaar.client;

import java.util.LinkedHashMap;
import java.util.Map;

import org.vcs.bazaar.client.commandline.CommandLineClient;
import org.vcs.bazaar.client.commandline.CommandLineClientFactory;
import org.vcs.bazaar.client.core.BazaarClientException;
import org.vcs.bazaar.client.xmlrpc.XmlRpcClientFactory;

/**
 * @author Guillermo Gonzalez
 * 
 */
public abstract class BazaarClientFactory {

	private static Map<String, BazaarClientFactory> factories = new LinkedHashMap<String, BazaarClientFactory>();
	static {
		registerAdapterFactory(new XmlRpcClientFactory());
		registerAdapterFactory(new CommandLineClientFactory());
	}

	public static IBazaarClient createClient(final String clientType) {
		if (factories.containsKey(clientType)) {
			final BazaarClientFactory factory = factories.get(clientType);
			try {
				if (factory.isAvailable(false)) {
					return (CommandLineClient) factory.createClientImpl();
				}
			} catch (BazaarClientException e) {
			}
		}
		return null;
	}

	public static String setupBestAvailableBackend(boolean force) throws BazaarClientException {
		BazaarClientException exc = null;
		for (BazaarClientFactory factory : factories.values()) {
			try {
				if (factory.isAvailable(force)) {
					return factory.getClientType();
				}
			} catch (BazaarClientException e) {
				exc = e;
			}
		}
		if (exc != null) {
			throw exc;
		}
		throw new BazaarClientException("No client factory found");
	}
	
	protected static void registerAdapterFactory(BazaarClientFactory newFactory) {
		factories.put(newFactory.getClientType(), newFactory);
	}
	
	public static String[] getClientTypes() {
		return factories.keySet().toArray(new String[0]);
	}

	public static String getPreferredClientType() throws BazaarClientException {
		return setupBestAvailableBackend(false);
	}

	public static final boolean isClientTypeAvailable(String clientType) {
		return createClient(clientType) != null;
	}

	protected abstract IBazaarClient createClientImpl();

	protected abstract String getClientType();

	protected abstract boolean isAvailable(boolean force) throws BazaarClientException;

}
