/**
 * LICENCSE + COPYRIGHT
 */
package org.vcs.bazaar.client;

import static org.vcs.bazaar.client.BazaarPreference.BZR_PROGRESS_BAR;
import static org.vcs.bazaar.client.BazaarPreference.EXECUTABLE;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.vcs.bazaar.client.commandline.CommandLineClientFactory;
import org.vcs.bazaar.client.core.BazaarClientException;
import org.vcs.bazaar.client.utils.BazaarUtilities;

/**
 * A singleton for handling some preferences and environment variables for bzr
 * executable
 *
 * @author Guillermo Gonzalez
 *
 * TODO: need test (and check if this are all the available env variables)
 */
public final class BazaarClientPreferences {

	private static final Log LOG = LogFactory.getLog(BazaarClientPreferences.class);

	private static final BazaarClientPreferences instance = new BazaarClientPreferences();

	private final HashMap<BazaarPreference, String> preferenceMap = new HashMap<BazaarPreference, String>();

	private String autodetectedExecutable;

	/**
	 * Default constructor which load some values for the environment (if they
	 * are set)
	 *
	 */
	private BazaarClientPreferences() {
		try {
			// do our best to find the executable
			File bzr = BazaarUtilities.findExecutable();
			autodetectedExecutable = bzr.getPath();
			preferenceMap.put(EXECUTABLE, bzr.getPath());
		} catch(FileNotFoundException e) {
			preferenceMap.put(EXECUTABLE, "bzr");
		}
		for (BazaarPreference pref : BazaarPreference.values()) {
			if(getSystemEnv(pref) != null) {
				preferenceMap.put(pref, getSystemEnv(pref));
			}
		}
	}

	/**
	 * Set all the elements corresponding to preferences from the {@link Properties}
	 * in the {@link BazaarClientPreferences} singleton
	 *
	 * Any value in the properties that is already setted is ignored.
	 *
	 * @param properties
	 */
	public void fillFrom(Properties properties) {
		for (BazaarPreference pref : BazaarPreference.values()) {
			if (preferenceMap.get(pref) == null) {
				preferenceMap.put(pref, getValue(properties, pref));
			}
		}
	}

	public void setFrom(Properties properties) {
		for (BazaarPreference pref : BazaarPreference.values()) {
			preferenceMap.put(pref, getValue(properties, pref));
		}
	}

	/**
	 * @return
	 */
	public final static BazaarClientPreferences getInstance() {
		return instance;
	}

	/**
	 * @param executable
	 * @return
	 */
	public String getString(BazaarPreference value) {
		return preferenceMap.get(value);
	}

	/**
	 *
	 * @param constantString
	 * @param value
	 */
	public final void set(BazaarPreference key, String value) {
		preferenceMap.put(key, value);
	}

	public final void unset(BazaarPreference key) {
		preferenceMap.remove(key);
	}

	/**
	 * @return the executable
	 */
	public final static List<String> getExecutable() {
		return getExecutable(true);
	}

	public final static List<String> getExecutable(boolean forceCLI) {

		String exe = getInstance().getString(EXECUTABLE);

		// If you aim at the std Bazaar distro DOS batch file,
		// we eat it and call Python direct (so we can kill it later)
		boolean isCLI = true;
		try {
			isCLI = forceCLI || BazaarClientFactory.getPreferredClientType().equals(CommandLineClientFactory.CLIENT_TYPE);
		} catch (BazaarClientException e) {
			LOG.info(e.getMessage());
		}
		if(exe.toLowerCase().endsWith(".bat") && isCLI) {
			try {
				return BazaarUtilities.getWindowsExecutable(exe);
			} catch(IOException e) {
				LOG.error("IO trouble while parsing bzr.bat", e);
			}
		}

		List<String> output = new ArrayList<String>();
		output.add(exe);
		return output;

	}

	/**
	 * returns all setted preferences (except EXECUTABLE)
	 *
	 * @return Map<String, String>
	 */
	public final Map<String, String> getPreferences() {
		final HashMap<String, String> newMap = new HashMap<String, String>(preferenceMap.size() - 1);
		for (BazaarPreference key : preferenceMap.keySet()) {
			if(!EXECUTABLE.equals(key) && !BZR_PROGRESS_BAR.equals(key) && preferenceMap.get(key) != null) {
                newMap.put(key.name(), preferenceMap.get(key));
            }
		}
		return newMap;
	}


	public final Map<BazaarPreference, String> asMap() {
	final HashMap<BazaarPreference, String> newMap = new HashMap<BazaarPreference, String>(preferenceMap.size() - 1);
		for (BazaarPreference key : preferenceMap.keySet()) {
			newMap.put(key, preferenceMap.get(key));
		}
		return newMap;
	}

	private static String getSystemEnv(BazaarPreference pref) {
		return getValue(null, pref);
	}

	private static String getValue(Properties properties, BazaarPreference pref) {
		if(properties != null) {
			return properties.getProperty(pref.toString(), System.getProperty(pref.toString()));
		}
		return System.getProperty(pref.toString());
	}

	public String getAutodetectedExecutable() {
		return autodetectedExecutable;
	}
}
