/**
 *
 */
package org.vcs.bazaar.client;

import static org.vcs.bazaar.client.utils.BazaarUtilities.unixFilePath;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;

/**
 * @author Guillermo Gonzalez
 *
 */
public class BazaarStatus implements IBazaarStatus {

	private static final long serialVersionUID = -7488117153422171224L;
	protected final EnumSet<BazaarStatusKind> statuses = EnumSet.noneOf(BazaarStatusKind.class);
	protected final File branchRoot;
	protected final File file;
	protected final boolean directory;

	public static IBazaarStatus create(File path, boolean directory, File branchRoot, BazaarStatusKind...status) {
		final List<BazaarStatusKind> statuses = new ArrayList<BazaarStatusKind>(1);
		for (BazaarStatusKind kind : status) {
			statuses.add(kind);
		}
		return new BazaarStatus(statuses, path, directory, branchRoot);
	}

	public BazaarStatus(File path, boolean directory, File branchRoot) {
		this.branchRoot = branchRoot;
		this.file = path;
		this.directory = directory;
	}

	public BazaarStatus(final List<BazaarStatusKind> statuses, File path, boolean directory, File branchRoot) {
		this(path, directory, branchRoot);
		this.statuses.addAll(statuses);
	}

	public boolean contains(BazaarStatusKind kind) {
		return statuses.contains(kind);
	}

	public String getAbsolutePath() {
		return unixFilePath(new File(getBranchRoot(), getPath()));
	}

	public File getBranchRoot() {
		return branchRoot;
	}

	public File getFile() {
		return file;
	}

	public String getNewKind() {
		return null;
	}

	public String getOldKind() {
		return null;
	}

	public String getPath() {
		if (file != null)
			return unixFilePath(file);
		else
			return "";
	}

	public File getPreviousFile() {
		return null;
	}

	public String getPreviousPath() {
		return null;
	}

	public String getShortStatus() {
		final StringBuilder versioned = new StringBuilder();
		final StringBuilder content = new StringBuilder();
		final StringBuilder execute = new StringBuilder();

		for (BazaarStatusKind kind : statuses) {
			if (kind.getCategory() == BazaarStatusKind.Category.VERSIONED) {
				versioned.append(kind.toChar());
			}
			if(kind.getCategory() == BazaarStatusKind.Category.CONTENT) {
				content.append(kind.toChar());
			}
			if(kind.getCategory() == BazaarStatusKind.Category.EXECUTABLE) {
				execute.append(kind.toChar());
			}
		}

		return versioned.append(content.toString()).append(execute.toString()).toString();
	}

	public Collection<? extends BazaarStatusKind> getStatuses() {
		return statuses;
	}

	@Override
	public String toString() {
		final StringBuilder sb = new StringBuilder();
		sb.append(getShortStatus()).append(" ");
		sb.append(getPath()).append(" ");
		return sb.toString();
	}

	public boolean isDirectory() {
		return directory;
	}

	@Override
	public boolean equals(Object obj) {
		if(obj == null) {
			return false;
		} else if (obj instanceof IBazaarStatus) {
			// this comparison is done only by path (bazaar spit out duplicated
			// status for one file i.e: when a file was modified and has
			// conflicts)
			boolean equalPath = ((IBazaarStatus) obj).getPath().equals(this.getPath());
			return equalPath && statuses.containsAll(((IBazaarStatus) obj).getStatuses());
		} else {
			return super.equals(obj);
		}

	}

	@Override
	public int hashCode() {
		final int PRIME = 31;
		int result = super.hashCode();
		result = PRIME * result + ((file == null) ? 0 : file.hashCode());
		result = PRIME * result + ((branchRoot == null) ? 0 : branchRoot.hashCode());
		return result;
	}
}
