package org.vcs.bazaar.client.commandline.internal;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Map;

import org.vcs.bazaar.client.BazaarClientPreferences;
import org.vcs.bazaar.client.BazaarPreference;
import org.vcs.bazaar.client.IBazaarPromptUserPassword;
import org.vcs.bazaar.client.commandline.parser.XMLParser;
import org.vcs.bazaar.client.core.BazaarClientException;

public abstract class CommandRunner {

	protected final BazaarClientPreferences preferences;

	private static HashMap<String, String> DEFAULT_ENVIROMENT = new HashMap<String, String>();

	static {
		// Here we put default env variables
		DEFAULT_ENVIROMENT.put(BazaarPreference.BZR_PROGRESS_BAR.name(), "xmlrpc"); //$NON-NLS-N$
		// if the executable is not configured, try to find it in the PATH
		// DEFAULT_ENVIROMENT.put("PATH", System.getenv("PATH"));
	}

	protected String stderr;
	protected String stdout;
	protected IBazaarPromptUserPassword userPasswordPrompt;

	public CommandRunner(BazaarClientPreferences preferences) {
		this.preferences = preferences;
	}

	public abstract void runCommand(final Command command, final File workDir) throws IOException,
			InterruptedException, BazaarClientException;

	/**
	 * @return the command stderr
	 */
	public String getStandardError() {
		return stderr;
	}

	public String getStandardOutput() {
		return stdout;
	}

	public String getStandardOutput(String charsetName) throws UnsupportedEncodingException {
	    return new String(stdout.getBytes(), charsetName);
	}

	public String[] getStandardOutputSplit() {
		String[] result = getStandardOutput().trim().split(getSplitExpression());
		if (result.length == 1 && result[0].isEmpty()) {
			result = new String[0];
		}
		return result;
	}

	protected abstract String getSplitExpression();

	public static void setDefaultEnviroment(ProcessBuilder pb, BazaarClientPreferences preferences) {
		Map<String, String> env = pb.environment();
		env.putAll(DEFAULT_ENVIROMENT);
		env.putAll(preferences.getPreferences());
	}

    /**
     * Detect declared XML encoding from a set of bytes. Because XML documents declare
     * their encodings (in the beginning and using ascii), we can read this information
     * to know how to decode the bytes. This is done for avoiding to decode from default 
     * system encoding, since it doesn't necessarily match the actual encoding of
     * the given bytes.
     * 
     * This method is available for implementors which receive bytes as response of the 
     * run commands, and where these bytes may be an XML document informing the encoding
     * of such bytes.
     * 
     * @param output the bytes to decode
     * @return the bytes decoded from the declared encoding, or from the default system 
     *         encoding if no encoding declaration is detected
     */
    protected String decodeXMLOutput(byte[] output) {
        if (output == null)
            return null;
        String decodedWithDefaultCharset = new String(output);
        String stringEncoding = XMLParser.getDeclaredEncoding(decodedWithDefaultCharset);
        if (stringEncoding == null)
            return decodedWithDefaultCharset;
        try {
            return new String(output, stringEncoding);
        } catch (UnsupportedEncodingException e) {
            return decodedWithDefaultCharset;
        }
    }

	public void setPasswordCallback(IBazaarPromptUserPassword callback) {
		this.userPasswordPrompt = callback;
	}

	public BazaarClientPreferences getPreferences() {
		return preferences;
	}

}
