/**
 *
 */
package org.vcs.bazaar.client.xmlrpc.internal;

import java.io.IOException;
import java.io.StringWriter;
import java.net.MalformedURLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;

import org.vcs.bazaar.client.BazaarClientPreferences;
import org.vcs.bazaar.client.BazaarPreference;
import org.vcs.bazaar.client.commandline.internal.CommandRunner;
import org.vcs.bazaar.client.core.BazaarClientException;
import org.vcs.bazaar.client.utils.StreamRedirect;
import org.vcs.bazaar.client.xmlrpc.XmlRpcCommandException;

import redstone.xmlrpc.XmlRpcArray;
import redstone.xmlrpc.XmlRpcException;
import redstone.xmlrpc.XmlRpcFault;

public class XMLRPCServer {

	private static final int MAX_RETRIES = 20;
	private static final int SLEEP_TIME = 200;
	private static final String START_XMLRPC = "start-xmlrpc";
	private static final String PORT_OPTION = "--port=";
	private static final String DEFAULT_PORT = "11111";
	private Process xmlRpcService;
	private final String port;
	private final BazaarClientPreferences preferences;
	private AtomicInteger runningCommands = new AtomicInteger(0);
	private AtomicInteger executionCount = new AtomicInteger(0);
	private final static Map<String, XMLRPCServer> INSTANCES = new ConcurrentHashMap<String, XMLRPCServer>();

	class ServerKiller extends Thread {
		public void run() {
			try {
				XMLRPCServer.this.shutdown(false);
			} catch (Exception ex) {
			}
		}
	}

	private XMLRPCServer(BazaarClientPreferences preferences, String port) {
		this.preferences = preferences;
		this.port = port;
		Runtime.getRuntime().addShutdownHook(new ServerKiller());
	}

	public synchronized void run() {
		try {
			if (testConnection()) {
				// there is an instance of the SERVICE already running
				return;
			}
		} catch (XmlRpcCommandException e) {
		}
		destroyProcess();
		executionCount.set(0);
		final List<String> args = new ArrayList<String>();
		args.addAll(BazaarClientPreferences.getExecutable(true));
		args.add(START_XMLRPC);
		if (this.port != null && !"".equals(this.port)) {
			args.add(PORT_OPTION + this.port);
		}
		final ProcessBuilder pb = createProcessBuilderWith(args);
		try {
			xmlRpcService = pb.start();
			final StreamRedirect srout = new StreamRedirect("xmlrpc-service_stdout>null",
					xmlRpcService.getInputStream(), null);
			final StringWriter serviceStdErr = new StringWriter();
			final StreamRedirect srerr = new StreamRedirect("xmlrpc-service_stderr>null",
					xmlRpcService.getErrorStream(), serviceStdErr);

			srout.start();
			srerr.start();

			boolean connect = isConnected();
			for (int i = 0; i < MAX_RETRIES && !connect; i++) {
				connect = isConnected();
				if (!connect) {
					try {
						xmlRpcService.exitValue();
						xmlRpcService = null;
						throw new BazaarClientException.BazaarUncheckedException(serviceStdErr.toString());
					} catch (IllegalThreadStateException e) {
						Thread.sleep(SLEEP_TIME);
					}
				}
			}
			if (!connect) {
				throw new BazaarClientException.BazaarUncheckedException("Could not connect to XMLRPC server");
			}
		} catch (IOException ioex) {
			throw new RuntimeException(ioex);
		} catch (InterruptedException intex) {
			throw new RuntimeException(intex);
		}
	}

	private boolean isConnected() throws InterruptedException {
		try {
			return testConnection();
		} catch (BazaarClientException ioex) {
			Thread.sleep(SLEEP_TIME);
		}
		return false;
	}

	private boolean testConnection() throws XmlRpcCommandException {
		try {
			final String result = (String) executeMethod("hello", new XmlRpcArray());
			if (result != null && result.equals("world!")) {
				return true;
			}
			return false;
		} catch (XmlRpcException e) {
			return false;
		}

	}

	private boolean isAlive() {
		try {
			if (!testConnection()) {
				if (xmlRpcService != null) {
					// if the process have exit value, it's finished
					xmlRpcService.exitValue();
				}
				return false;
			}
			return true;
		} catch (IllegalThreadStateException e) {
			// if the Process object has not yet terminated.
			return true;
		} catch (XmlRpcCommandException e) {
			return false;
		}
	}

	public synchronized void shutdown(boolean force) {
		try {
			if (xmlRpcService != null || force) {
				executeMethod("quit", new XmlRpcArray());
			}
		} catch (Exception ex) {
		} finally {
			destroyProcess();
		}
	}

	private void destroyProcess() {
		try {
			if (xmlRpcService != null) {
				xmlRpcService.destroy();
			}
		} finally {
			xmlRpcService = null;
		}
	}

	/**
	 * Executes a xmlrpc method.
	 * 
	 * @param method
	 *            the method's name
	 * @param params
	 *            a {@link XmlRpcArray} representing the method arguments
	 * @return the result of executing the method
	 * @throws XmlRpcCommandException
	 */
	public Object executeMethod(String method, XmlRpcArray params) throws XmlRpcCommandException {
		try {
			return getClient(false).invoke(method, params);
		} catch (XmlRpcFault e) {
			throw XmlRpcCommandException.wrapException(e);
		} catch (MalformedURLException e) {
			throw XmlRpcCommandException.wrapException(e);
		}
	}

	public redstone.xmlrpc.XmlRpcClient getClient(boolean secondary) throws MalformedURLException {
		String port = this.port;
		if (secondary) {
			port = String.valueOf(Integer.parseInt(port) + 1);
		}
		return new redstone.xmlrpc.XmlRpcClient("http://127.0.0.1:" + port + "/RPC2", false);
	}

	/**
	 * Instantiate a new ProcessBuilder and modify it's enviroment variables to
	 * be suitable to launch a bzr process.
	 * 
	 * @param args
	 * @return
	 */
	public ProcessBuilder createProcessBuilderWith(List<String> args) {
		final ProcessBuilder pb = new ProcessBuilder(args);
		CommandRunner.setDefaultEnviroment(pb, preferences);
		return pb;
	}

	/**
	 * Check the SERVICE status (running/port changed) and restart it if
	 * necessary
	 */
	public synchronized void checkStatus() {
		if (!isAlive()) {
			run();
		}
	}

	private int getRestartThreshold() {
		try {
			String t = preferences.getString(BazaarPreference.BZR_XMLRPC_RESTART_THRESHOLD);
			if (t != null) {
				return Integer.parseInt(t);
			}
		} catch (Exception e) {
		}
		return 0;
	}

	public synchronized void startCommand() {
		int restartThreshold = getRestartThreshold();
		if (restartThreshold > 0 && executionCount.incrementAndGet() > restartThreshold) {
			while (runningCommands.get() > 0) {
				try {
					Thread.sleep(100);
				} catch (InterruptedException e) {
				}
			}
			shutdown(true);
			for (int i = 0; i < MAX_RETRIES; i++) {
				try {
					checkStatus();
					break;
				} catch (Exception e) {
					try {
						Thread.sleep(SLEEP_TIME);
					} catch (InterruptedException e1) {
					}
				}
			}
			executionCount.addAndGet(-restartThreshold);
		}
		runningCommands.incrementAndGet();
	}

	public void endCommand() {
		runningCommands.decrementAndGet();
	}

	private static String getPort(BazaarClientPreferences preferences) {
		String p = preferences.getString(BazaarPreference.BZR_XMLRPC_PORT);
		if (p == null || p.isEmpty()) {
			p = DEFAULT_PORT;
		}
		return p;
	}

	public static synchronized XMLRPCServer getInstance(BazaarClientPreferences preferences) {
		String port = getPort(preferences);
		XMLRPCServer server = INSTANCES.get(port);
		if (server == null) {
			server = new XMLRPCServer(preferences, port);
			INSTANCES.put(port, server);
		}
		return server;
	}
}