/**
 * LICENCSE + COPYRIGHT
 */
package org.vcs.bazaar.client;

import java.util.Arrays;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.vcs.bazaar.client.core.BazaarClientException;
import org.vcs.bazaar.client.testUtils.BazaarTest;
import org.vcs.bazaar.client.testUtils.Environment;

/**
 * @author Piotr Piastucki
 */
public class FrequentRestartTest extends BazaarTest {

	@Before
	public void setup() {
		BazaarClientPreferences prefs = BazaarClientPreferences.getInstance();
		prefs.set(BazaarPreference.BZR_XMLRPC_RESTART_THRESHOLD, "2");
	}

	@After
	public void cleanup() {
		BazaarClientPreferences prefs = BazaarClientPreferences.getInstance();
		prefs.unset(BazaarPreference.BZR_XMLRPC_RESTART_THRESHOLD);
	}

	@Test
	public final void testRestart() throws Exception {
		Environment testEnv = new Environment("restart", getTestConfig());
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		for (int i = 0; i < 10; i++) {
			testLog(testEnv);
			testNick(testEnv);
			testRevno(testEnv);
		}
	}

	@Test
	public final void testMultiThreadRestart() throws Exception {
		final Environment testEnv = new Environment("restart", getTestConfig());
		client.setWorkDir(testEnv.getWorkingTreeLocation());
		final AtomicBoolean stop = new AtomicBoolean(false);
		final int commands[] = new int[4];
		Arrays.fill(commands, 0);
		final AtomicBoolean error = new AtomicBoolean(false);
		Thread t1 = new Thread(new Runnable() {
			public void run() {
				while (!stop.get()) {
					try {
						testLog(testEnv);
					} catch (Throwable e) {
						error.set(true);
					} finally {
						commands[0]++;
					}
				}
			}
		});
		Thread t2 = new Thread(new Runnable() {
			public void run() {
				while (!stop.get()) {
					try {
						testLog(testEnv);
					} catch (Throwable e) {
						error.set(true);
					} finally {
						commands[1]++;
					}
				}
			}
		});
		Thread t3 = new Thread(new Runnable() {
			public void run() {
				while (!stop.get()) {
					try {
						testNick(testEnv);
					} catch (Throwable e) {
						error.set(true);
					} finally {
						commands[2]++;
					}
				}
			}
		});
		Thread t4 = new Thread(new Runnable() {
			public void run() {
				while (!stop.get()) {
					try {
						testRevno(testEnv);
					} catch (Throwable e) {
						error.set(true);
					} finally {
						commands[3]++;
					}
				}
			}
		});
		t1.start();
		t2.start();
		t3.start();
		t4.start();
		boolean finished = false;
		while (!finished) {
			finished = true;
			for (int i : commands) {
				if (i < 5) {
					finished = false;
					break;
				}
			}
			Thread.sleep(50);
		}
		stop.set(true);
		t1.join();
		t2.join();
		t3.join();
		t4.join();
		Assert.assertFalse(error.get());
	}

	private void testRevno(Environment testEnv) throws BazaarClientException {
		BazaarRevision expectedRevision = BazaarRevision.getRevision(1);
		testEnv.getExpectedWorkingTree().setItemsRevision(expectedRevision);
		BazaarRevision result = client.revno(testEnv.getWorkingTreeLocation());
		Assert.assertEquals(Integer.valueOf(expectedRevision.getValue().trim()),
				Integer.valueOf(result.getValue().trim()));
	}

	private void testNick(Environment testEnv) throws BazaarClientException {
		String nick = client.nick(null);
		Assert.assertEquals(testEnv.getWorkingTreeLocation().getName().trim(), nick);
	}

	private void testLog(Environment testEnv) throws BazaarClientException {
		List<IBazaarLogMessage> logMsgs = client.log(testEnv.getWorkingTreeLocation());
		Assert.assertEquals(1, logMsgs.size());
		for (IBazaarLogMessage message : logMsgs) {
			Assert.assertEquals("default_branch", message.getBranchNick());
			Assert.assertEquals(BazaarRevision.getRevision(1).getValue(), message.getRevision().getValue());
			Assert.assertEquals(getUserName(), message.getCommiter());
			Assert.assertEquals("initial import", message.getMessage().trim());
		}
	}
}
