/**
 *
 */
package org.vcs.bazaar.client.xmlrpc.internal;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.junit.Before;
import org.junit.Test;
import org.vcs.bazaar.client.BazaarClientPreferences;
import org.vcs.bazaar.client.IBzrError;
import org.vcs.bazaar.client.IPlugin;
import org.vcs.bazaar.client.commandline.CommandLineException;
import org.vcs.bazaar.client.commandline.commands.Plugins;
import org.vcs.bazaar.client.commandline.commands.Version;
import org.vcs.bazaar.client.commandline.internal.Command;
import org.vcs.bazaar.client.commandline.internal.ShellCommandRunner;
import org.vcs.bazaar.client.commandline.parser.XMLPluginParser;
import org.vcs.bazaar.client.core.BazaarClientException;
import org.vcs.bazaar.client.testUtils.BazaarTest;
import org.vcs.bazaar.client.testUtils.Environment;
import org.vcs.bazaar.client.utils.BazaarUtilities;
import org.vcs.bazaar.client.xmlrpc.XmlRpcCommandException;
import org.vcs.bazaar.client.xmlrpc.XmlRpcMethod;

import redstone.xmlrpc.XmlRpcArray;
import redstone.xmlrpc.XmlRpcStruct;

/**
 * @author Guillermo Gonzalez <guillo.gonzo AT gmail DOT com>
 * @author Renato Silva
 *
 */
public class XMLRPCCommandRunnerTest extends BazaarTest {

	private XMLRPCCommandRunner xmlRpcRunner;
	private ShellCommandRunner shellRunner;

	@Before
	public void configureRunners() {
		xmlRpcRunner = new XMLRPCCommandRunner(XMLRPCServer.getInstance(BazaarClientPreferences.getInstance()));
		shellRunner = new ShellCommandRunner();
	}

	/**
	 * Test method for {@link org.vcs.bazaar.client.xmlrpc.internal.XMLRPCCommandRunner#runCommand(java.util.List, java.io.File)}.
	 * @throws BazaarClientException
	 */
	@Test
	public void testRunCommand() throws BazaarClientException {
		Command cmd = new Version();
		xmlRpcRunner.runCommand(cmd, getTempDir());

		Integer exitVal = xmlRpcRunner.result.getIntegerWrapper(0);
		String stdOut = new String(xmlRpcRunner.result.getBinary(1));
		String stdErr = xmlRpcRunner.result.getString(2);
		assertNotNull(exitVal);
		assertEquals(Integer.valueOf(0), exitVal);
		assertNotNull(stdOut);
		assertNotNull(stdErr);
		assertEquals("", stdErr);
	}

	@Test
	public void testRunCommandWithError() throws BazaarClientException  {
		Command cmd = new Command() {
			
			@Override
			public String getCommand() {
				return "versionxml";
			}
			
			@Override
			protected List<String> getArguments() throws CommandLineException {
				return new ArrayList<String>();
			}
		};
		try {
			xmlRpcRunner.runCommand(cmd, getTempDir());
		} catch (XmlRpcCommandException e) {
			IBzrError error = e.getCommandError();
			assertEquals(error.getType(), "BzrCommandError");
			assertEquals(error.getMessage(), "unknown command \"versionxml\"");
		}
	}

	@Test
	public void testExecuteMethod() throws BazaarClientException {
		XmlRpcMethod method = new XmlRpcMethod("hello");
		Object result = xmlRpcRunner.executeMethod(method);
		assertNotNull(result);
		assertEquals(result, "world!");
	}

	@SuppressWarnings("unchecked")
    @Test
	public void testSearch() throws Exception {
		if(!checkSearchPlugin()) {
			// if bzr-search isn't installed ignore the test
			return;
		}
		Environment env = new Environment("xmlrpc-search", getTestConfig());

		Command cmd = new Command() {
			
			@Override
			public String getCommand() {
				return "index";
			}
			
			@Override
			protected List<String> getArguments() throws CommandLineException {
				return new ArrayList<String>();
			}
		};
		final File workDir = env.getWorkingTreeLocation();
		shellRunner.runCommand(cmd, workDir);

		XmlRpcMethod method = new XmlRpcMethod("search", BazaarUtilities.unixFilePath(env.getWorkingTreeLocation()), new String[]{"a", "file", "in"});
		Object retVal = xmlRpcRunner.executeMethod(method);

		assertEquals(retVal.getClass(), XmlRpcStruct.class);
		XmlRpcStruct result = (XmlRpcStruct)retVal;
		assertNotNull(result);
		List<Map<String, String>> fileHits = result.getArray("file_hits");
		List<Map<String, String>> revIdHits = result.getArray("revid_hits");
		List<String> pathHits = result.getArray("path_hits");
		assertNotNull(revIdHits);
		assertNotNull(fileHits);
		assertNotNull(pathHits);
		assertNull(method.getError());
		assertEquals("Expected file hits", 6, fileHits.size());
	}

    @Test
    public void testGetStandardOutput() throws UnsupportedEncodingException, BazaarClientException {
        testOutput(OutputType.STANDARD);
    }

    @Test
    public void testGetStandardError() throws UnsupportedEncodingException, BazaarClientException {
        testOutput(OutputType.ERROR);
    }

    private enum OutputType { STANDARD, ERROR };

    private String getOutput(Command command, OutputType type) {
        switch (type) {
        case STANDARD:
            return command.getStandardOutput();
        case ERROR:
            return command.getStandardError();
        default:
            return null;
        }
    }

    private String sampleXML(String encoding, String content) {
        return String.format("<?xml version=\"1.0\" encoding=\"%s\"?><hello>%s</hello>", encoding, content);
    }

    private void testOutput(OutputType outputType) throws BazaarClientException, UnsupportedEncodingException {

        final String NON_XML = "Raw";
        final String NON_ASCII = "Maçã";
        final String NON_ASCII_DECODED_FROM_UTF8_BYTES_AS_LATIN1 = "MaÃ§Ã£";

        final DummyCommand dummyCommand = new DummyCommand();
        final DummyCommandRunner dummyRunner = new DummyCommandRunner();


        // If output is not XML, then should just return it.
        dummyCommand.execute(dummyRunner);
        dummyRunner.setDummyOutput(NON_XML.getBytes("UTF-8"), NON_XML);
        assertEquals(NON_XML, getOutput(dummyCommand, outputType));


        // If output is XML, should detect declared encoding and decode printed bytes from such encoding.
        // The encoding declared in the XML header is assumed to be the correct.

        final String declaredAsUTF8 = sampleXML("UTF-8", NON_ASCII);
        dummyRunner.setDummyOutput(declaredAsUTF8.getBytes("UTF-8"), declaredAsUTF8);
        dummyCommand.execute(dummyRunner);
        assertEquals("Ouptut bytes should be decoded as UTF-8", declaredAsUTF8, getOutput(dummyCommand, outputType));

        final String declaredAsLatin1 = sampleXML("ISO-8859-1", NON_ASCII);
        final String utf8DecodedAsLatin1 = declaredAsLatin1.replaceAll(NON_ASCII, NON_ASCII_DECODED_FROM_UTF8_BYTES_AS_LATIN1);
        dummyRunner.setDummyOutput(declaredAsLatin1.getBytes("UTF-8"), utf8DecodedAsLatin1);
        dummyCommand.execute(dummyRunner);
        assertEquals("Output bytes should be UTF-8 decoded as Latin-1", utf8DecodedAsLatin1, getOutput(dummyCommand, outputType));
    }

	private boolean checkSearchPlugin() throws BazaarClientException {
		Plugins cmd = new Plugins();
		cmd.execute(xmlRpcRunner);
		for (IPlugin plugin : new XMLPluginParser().parse(cmd.getStandardOutput())) {
			if(plugin.getName() != null && plugin.getName().contains("search")) {
				return true;
			}
		}
		return false;
	}

	private File getTempDir() {
		return new File(System.getProperty("java.io.tmpdir"));
	}

}


class DummyCommandRunner extends XMLRPCCommandRunner {

    @SuppressWarnings("unchecked")
    public DummyCommandRunner() {
    	super(XMLRPCServer.getInstance(BazaarClientPreferences.getInstance()));
        result = new XmlRpcArray();
        result.add(0, null);
        result.add(1, null);
        result.add(2, null);
    }

    @SuppressWarnings("unchecked")
    public void setDummyOutput(byte[] dummyStandardOutput, String dummyErrorOutput) throws UnsupportedEncodingException {
        result.set(1, dummyStandardOutput);
        result.set(2, dummyErrorOutput);
    }

    @Override
    public void runCommand(Command cmd, File workDir) throws BazaarClientException {
        // Dummy command, do nothing
    }

}


class DummyCommand extends Command {

    @Override
    protected List<String> getArguments() throws CommandLineException {
        return new ArrayList<String>();
    }

    @Override
    public String getCommand() {
        return "dummy-command";
    }

}
