# -*- encoding: utf-8 -*-

from bzrlib import log
from bzrlib.lazy_import import lazy_import
lazy_import(globals(), """
import bzrlib
from bzrlib import (
    debug,
    osutils,
    )
""")

from writer import _escape_cdata


class XMLLogFormatter(log.LogFormatter):
    """ add a --xml format to 'bzr log'"""

    supports_merge_revisions = True
    supports_delta = True
    supports_tags = True

    def __init__(self, *args, **kwargs):
        super(XMLLogFormatter, self).__init__(*args, **kwargs)
        self.log_count = 0
        self.start_with_merge = False
        self.previous_merge_depth = 0
        self.debug_enabled = 'debug' in debug.debug_flags
        self.stack = []

    def show(self, revno, rev, delta, tags=None):
        lr = log.LogRevision(rev, revno, 0, delta, tags)
        return self.log_revision(lr)

    def show_merge_revno(self, rev, merge_depth, revno):
        """a call to self._show_helper, XML don't care about formatting """
        lr = log.LogRevision(rev, merge_depth=merge_depth, revno=revno)
        return self.log_revision(lr)

    def log_revision(self, revision):
        """Log a revision, either merged or not."""
        to_file = self.to_file
        if self.debug_enabled:
            self.__debug(revision)
        actions = []
        # to handle merge revision as childs
        if self.previous_merge_depth < revision.merge_depth:
            if self.log_count > 0:
                self.__open_merge(revision.merge_depth - self.previous_merge_depth)
        elif self.previous_merge_depth > revision.merge_depth:
            self.__close_merge(self.previous_merge_depth - revision.merge_depth)
            if len(self.stack) > 0 and self.stack[len(self.stack) - 1] == 'log':
                self.__close_log()
        else:        
            self.__close_log()
        self.__open_log()
        self.__log_revision(revision)

        self.log_count = self.log_count + 1
        self.previous_merge_depth = revision.merge_depth

    def __open_merge(self, levels):
        for i in range(levels):
            self.to_file.write('<merge>')
            self.stack.append('merge')

    def __open_log(self):
        self.to_file.write('<log>',)
        self.stack.append('log')

    def __close_merge(self, levels):
        while len(self.stack) > 0:
            item = self.stack.pop(len(self.stack) - 1)
            self.to_file.write('</%s>' % item)
            if item == 'merge':
                levels -= 1
                if levels == 0:
                    return
            
    def __close_log(self):
        while len(self.stack) > 0:
            item = self.stack.pop(len(self.stack) - 1)
            self.to_file.write('</%s>' % item)
            if item == 'log':
                return

    def __log_revision(self, revision):
        if revision.revno is not None:
            self.to_file.write('<revno>%s</revno>' % revision.revno)
        if revision.tags:
            self.to_file.write('<tags>')
            for tag in revision.tags:
                self.to_file.write('<tag>%s</tag>' % _escape_cdata(tag))
            self.to_file.write('</tags>')
        if self.show_ids:
            self.to_file.write('<revisionid>%s</revisionid>' %
                                revision.rev.revision_id)
            if len(revision.rev.parent_ids) > 0:
                self.to_file.write('<parents>')
            for parent_id in revision.rev.parent_ids:
                self.to_file.write('<parent>%s</parent>' % parent_id)
            if len(revision.rev.parent_ids) > 0:
                self.to_file.write('</parents>')

        self.to_file.write('<committer>%s</committer>' % \
                        _escape_cdata(revision.rev.committer))

        authors = revision.rev.get_apparent_authors()
        if authors != [revision.rev.committer]:
            self.to_file.write('<authors>')
            for a in authors:
                self.to_file.write('<author>%s</author>' % _escape_cdata(a))
            self.to_file.write('</authors>')

        try:
            self.to_file.write('<branch-nick>%s</branch-nick>' % \
                _escape_cdata(revision.rev.properties['branch-nick']))
        except KeyError:
            pass
        date_str = osutils.format_date(revision.rev.timestamp,
                               revision.rev.timezone or 0,
                               self.show_timezone)
        self.to_file.write('<timestamp>%s</timestamp>' % date_str)

        self.to_file.write('<message><![CDATA[')
        if not revision.rev.message:
            self.to_file.write('(no message)')
        else:
            self.to_file.write(_format_message(revision.rev.message))
        self.to_file.write(']]></message>')
        if revision.delta is not None:
            from statusxml import show_tree_xml
            self.to_file.write('<affected-files>')
            show_tree_xml(revision.delta, self.to_file, self.show_ids)
            self.to_file.write('</affected-files>')

    def begin_log(self):
        self.to_file.write('<?xml version="1.0" encoding="%s"?>' % \
                osutils.get_user_encoding())
        self.to_file.write('<logs>')

    def end_log(self):
        #if the last logged was inside a merge (and it was only one log)
        self.__close_merge(len(self.stack))
        self.to_file.write('</logs>')


class XMLLineLogFormatter(log.LineLogFormatter):

    def __init__(self, *args, **kwargs):
        super(XMLLineLogFormatter, self).__init__(*args, **kwargs)

    def log_string(self, revno, rev, max_chars=50):
        """Format log info into one string style. Don't truncate the string
        like LineLogFormatter because we are writting xml, and don't make sense
        to truncate the string.
        :param  revno:      revision number (int) or None.
                            Revision numbers counts from 1.
        :param  rev:        revision info object
        :return:            formatted truncated string
        """
        out = []
        out.append('<log>')
        if revno:
            # show revno only when is not None
            out.append("<revno>%s</revno>" % revno)
        elif rev.revision_id:
            out.append("<revisionid>%s</revisionid>" % rev.revision_id)
        out.append('<committer>%s</committer>' %
                   _escape_cdata(rev.committer))
        date_str = osutils.format_date(rev.timestamp,
                            rev.timezone or 0,
                            show_offset=True)
        out.append('<timestamp>%s</timestamp>' % date_str)

        out.append('<message><![CDATA[%s]]></message>' % \
                   _format_message(rev.message))
        out.append('</log>')
        return " ".join(out).rstrip('\n')


def line_log(rev):
    lf = XMLLineLogFormatter(None)
    return lf.log_string(None, rev)


def _format_message(rev_message):
    return rev_message.rstrip('\n')
