# Copyright (C) 2005, 2006, 2007, 2008, 2009 Canonical Ltd
# Copyright (C) 2013 Piotr Piastucki
# -*- coding: utf-8 -*-
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


"""Tests for bzr xmlconflicts."""

import os
import sys
import bzrlib
from bzrlib import osutils
from bzrlib.tests.blackbox import ExternalBase
from bzrlib.tests import TestCaseInTempDir, TestCaseWithTransport
from bzrlib.xml_serializer import elementtree as elementtree
fromstring = elementtree.fromstring
elementtree_tostring = elementtree.tostring

class TestConflicts(ExternalBase):

    def _prepare(self):
        # This scenario will create:
        # 2 text conflicts
        # 1 content conflict (file renamed-modified/deleted)
        # 1 path conflict (renamed/deleted) 
        # 1 duplicate entry conflict (file created with different IDs)
        base_actions = [('add', ('file', 'file-id-1', 'file', 'trunk content\n')),
                        ('add', ('dir', 'dir-id', 'directory', '')),
                        ('add', ('dir/file', 'file-id-2', 'file', 'trunk content\n')),
                        ('add', ('content-conflict', 'file-id-3', 'file', 'trunk content\n')),
                        ('add', ('path-conflict', 'file-id-4', 'file', 'trunk content\n'))
                       ]
        other_actions = [('modify', ('file-id-1', 'trunk content\nfeature A\n')),
                         ('modify', ('file-id-2', 'trunk content\nfeature A\n')),
                         ('modify', ('file-id-3', 'trunk content\nmore content\n')),
                         ('unversion', 'file-id-4'),
                         ('add', ('duplicate-conflict', 'file-id-5', 'file', 'trunk content\n'))
                        ]

        this_actions = [('modify', ('file-id-1', 'trunk content\nfeature B\n')),
                        ('modify', ('file-id-2', 'trunk content\nfeature B\n')),
                        ('unversion', 'file-id-3'),
                        ('rename', ('path-conflict', 'path-conflict-new')),
                        ('add', ('duplicate-conflict', 'file-id-6', 'file', 'trunk content\n'))
                        ]

        builder = self.make_branch_builder('trunk')
        builder.start_series()
        # Create an empty trunk
        builder.build_snapshot('start', None, [
                ('add', ('', 'root-id', 'directory', ''))])
        # Add a minimal base content
        builder.build_snapshot('base', ['start'], base_actions)
        # Modify the base content in branch
        builder.build_snapshot('other', ['base'], other_actions)
        # Modify the base content in trunk
        builder.build_snapshot('this', ['base'], this_actions)
        # builder.get_branch() tip is now 'this'

        builder.finish_series()
        b = builder.get_branch()
        wt = b.bzrdir.sprout('branch').open_workingtree()
        wt.merge_from_branch(b, 'other')
        return wt

    def validate(self, conflicts_xml, expected):
        for conflict in conflicts_xml:
            self.assertEquals(conflict[0].tag, 'type') 
            self.assertEquals(conflict[1].tag, 'path') 
            self.assertEquals(conflict[2].tag, 'message') 
            found = False
            for i in range(0, len(expected)):
                if expected[i][0] == conflict[0].text and \
                   expected[i][1] == conflict[1].text:
                    expected.remove(expected[i])
                    found = True
                    break
            self.assertEquals(found, True) 

    def test_text_conflicts(self):
        self._prepare()
        conflicts_xml = fromstring(self.run_bzr('xmlconflicts --text --directory=branch')[0])
        self.assertEquals(conflicts_xml.tag, 'conflicts') 
        self.assertEquals(len(conflicts_xml), 2)
        expected=[('text conflict', 'file'), 
                  ('text conflict', 'dir/file')]
        self.validate(conflicts_xml, expected)

    def test_mixed_conflicts(self):
        self._prepare()
        conflicts_xml = fromstring(self.run_bzr('xmlconflicts --directory=branch')[0])
        self.assertEquals(conflicts_xml.tag, 'conflicts') 
        self.assertEquals(len(conflicts_xml), 5)
        expected=[('text conflict', 'file'), 
                  ('text conflict', 'dir/file'),
                  ('contents conflict', 'content-conflict'),
                  ('path conflict', 'path-conflict-new'),
                  ('duplicate', 'duplicate-conflict.moved'),
]
        self.validate(conflicts_xml, expected)

    def test_empty(self):
        conflicts_xml = fromstring(self.run_bzr('xmlconflicts')[0])
        self.assertEquals(conflicts_xml.tag, 'conflicts') 
        self.assertEquals(len(conflicts_xml), 0)

    def test_conflicts_outside_branch(self):
        # we disable isolation because the error we want to catch is triggered
        # outside of the jail
        self.disable_directory_isolation()
        if sys.platform == "win32":
            location = "C:/i/do/not/exist/"
        else:
            location = "/i/do/not/exist/"
        out, err = self.run_bzr('xmltags --directory='+ location, retcode=3)
        self.assertEqual(out, '')
        self.assertEqual(
            '<?xml version="1.0" encoding="%s"?><error>'
            '<class>NotBranchError</class><dict><key>path</key><value>'
            '%s</value><key>extra</key><value></value>'
            '<key>detail</key><value></value></dict>'
            '<message>Not a branch: "%s".</message>'
            '</error>' % (osutils.get_user_encoding(),
                          location, location), err)


