# Copyright (C) 2005, 2006, 2007, 2008, 2009 Canonical Ltd
# Copyright (C) 2013 Piotr Piastucki
# -*- coding: utf-8 -*-
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


"""Black-box tests for bzr shelve list."""

import os
import sys
import bzrlib
from bzrlib import osutils
from bzrlib import shelf
from bzrlib.tests.blackbox import ExternalBase
from bzrlib.tests import TestCaseInTempDir, TestCaseWithTransport
from bzrlib.xml_serializer import elementtree as elementtree
fromstring = elementtree.fromstring
elementtree_tostring = elementtree.tostring

class TestShelveList(ExternalBase):

    def prepare_shelves(self):
        tree = self.make_branch_and_tree('.')
        tree.lock_write()
        self.addCleanup(tree.unlock)
        self.build_tree(['foo/', 'bar/', 'foo/baz', 'foo/text'])
        self.build_tree_contents([('foo/text', 'a\n')])
        tree.add(['foo', 'bar', 'foo/baz', 'foo/text'], ['foo-id', 'bar-id', 'baz-id', 'text-id'])
        tree.commit('Committed foo')
        self.build_tree_contents([('foo/text', 'b\na\nc\n')])
        # shelf #1 - content change
        creator = shelf.ShelfCreator(tree, tree.basis_tree())
        creator.shelve_all()
        manager = tree.get_shelf_manager()
        manager.shelve_changes(creator, 'message1')
        creator.finalize()
        tree.rename_one('foo/baz', 'bar/baz')
        # shelf #2 - rename
        creator = shelf.ShelfCreator(tree, tree.basis_tree())
        creator.shelve_all()
        manager.shelve_changes(creator, 'message2')
        creator.finalize()
        return tree

    def test_list_all(self):
        tree = self.prepare_shelves()
        shelves_xml = fromstring(self.run_bzr('xmlshelvelist')[0])
       # tree = WorkingTree.open_containing('.')[0]
        manager = tree.get_shelf_manager()
        shelves = manager.active_shelves()
        self.assertEquals(shelves_xml.tag, 'shelves') 
        self.assertEquals(len(shelves_xml), 2)
        self.assertEquals(shelves_xml[0].tag, 'shelf')
        self.assertEquals(shelves_xml[0][0].tag, 'id') 
        self.assertEquals(shelves_xml[0][0].text, '2') 
        self.assertEquals(shelves_xml[0][1].tag, 'message') 
        self.assertEquals(shelves_xml[0][1].text, 'message2') 
        self.assertEquals(shelves_xml[1].tag, 'shelf')
        self.assertEquals(shelves_xml[1][0].tag, 'id') 
        self.assertEquals(shelves_xml[1][0].text, '1') 
        self.assertEquals(shelves_xml[1][1].tag, 'message') 
        self.assertEquals(shelves_xml[1][1].text, 'message1') 

    def test_empty(self):
        shelves_xml = fromstring(self.run_bzr('xmlshelvelist')[0])
        self.assertEquals(shelves_xml.tag, 'shelves') 
        self.assertEquals(len(shelves_xml), 0)

    def test_shelve_list_outside_branch(self):
        # we disable isolation because the error we want to catch is triggered
        # outside of the jail
        self.disable_directory_isolation()
        if sys.platform == "win32":
            location = "C:/i/do/not/exist/"
        else:
            location = "/i/do/not/exist/"
        out, err = self.run_bzr('xmlshelvelist --directory='+ location, retcode=3)
        self.assertEqual(out, '')
        self.assertEqual(
            '<?xml version="1.0" encoding="%s"?><error>'
            '<class>NotBranchError</class><dict><key>path</key><value>'
            '%s</value><key>extra</key><value></value>'
            '<key>detail</key><value></value></dict>'
            '<message>Not a branch: "%s".</message>'
            '</error>' % (osutils.get_user_encoding(),
                          location, location), err)


