# Copyright (C) 2005, 2006, 2007, 2008, 2009 Canonical Ltd
# Copyright (C) 2013 Piotr Piastucki
# -*- coding: utf-8 -*-
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


"""Black-box tests for bzr tags."""

import os
import sys
import bzrlib
from bzrlib import osutils
from bzrlib.tests.blackbox import ExternalBase
from bzrlib.tests import TestCaseInTempDir, TestCaseWithTransport
from bzrlib.xml_serializer import elementtree as elementtree
fromstring = elementtree.fromstring
elementtree_tostring = elementtree.tostring

class TestTags(ExternalBase):

    def _prepare(self, path='.', tagsufix = '', format=None):
        tree = self.make_branch_and_tree(path, format=format)
        self.build_tree(
            [path + '/hello.txt', path + '/goodbye.txt', path + '/meep.txt'])
        tree.add('hello.txt')
        tree.commit(message='message1', rev_id='revid-1')
        tree.add('goodbye.txt')
        tree.commit(message='message2', rev_id='revid-2')
        tree.add('meep.txt')
        tree.commit(message='message3', rev_id='revid-3')
        b = tree.branch
        b.tags.set_tag(u'tag-1' + tagsufix, 'revid-1')
        b.tags.set_tag(u'tag-1.1' + tagsufix, 'revid-1')
        b.tags.set_tag(u'tag-2' + tagsufix, 'revid-3')
        return tree

    def test_all_tags(self):
        self._prepare()
        tags = [u'tag-1', u'tag-1.1', u'tag-2']
        revisions = ['1', '1', '3']
        tags_xml = fromstring(self.run_bzr('xmltags')[0])
        self.assertEquals(tags_xml.tag, 'tags') 
        i = 0
        for elem1 in list(tags_xml):
            self.assertEquals(elem1.tag, 'tag')
            self.assertEquals(elem1[0].tag, 'name') 
            self.assertEquals(elem1[1].tag, 'revision') 
            self.assertEquals(elem1[0].text, tags[i]) 
            self.assertEquals(elem1[1].text, revisions[i]) 
            i += 1

    def test_empty(self):
        tags_xml = fromstring(self.run_bzr('xmltags')[0])
        self.assertEquals(tags_xml.tag, 'tags') 
        self.assertEquals(len(tags_xml), 0)

    def test_tags_outside_branch(self):
        # we disable isolation because the error we want to catch is triggered
        # outside of the jail
        self.disable_directory_isolation()
        if sys.platform == "win32":
            location = "C:/i/do/not/exist/"
        else:
            location = "/i/do/not/exist/"
        out, err = self.run_bzr('xmltags --directory='+ location, retcode=3)
        self.assertEqual(out, '')
        self.assertEqual(
            '<?xml version="1.0" encoding="%s"?><error>'
            '<class>NotBranchError</class><dict><key>path</key><value>'
            '%s</value><key>extra</key><value></value>'
            '<key>detail</key><value></value></dict>'
            '<message>Not a branch: "%s".</message>'
            '</error>' % (osutils.get_user_encoding(),
                          location, location), err)

    def test_merged_tags(self):
        branch1_tree = self._prepare(path='branch1')
        branch1 = branch1_tree.branch
        branch2_tree = branch1_tree.bzrdir.sprout('branch2').open_workingtree()
        branch2 = branch2_tree.branch
        branch1.tags.set_tag('tag-4', branch1.last_revision())
        branch1_tree.commit(message='foobar', allow_pointless=True)
        os.chdir('branch2')
        tags_xml = fromstring(self.run_bzr("xmltags")[0])
        self.assertEquals(len(tags_xml), 3)
        branch2_tree.merge_from_branch(branch1)
        branch1.tags.merge_to(branch2.tags)
        tags_xml = fromstring(self.run_bzr("xmltags")[0])
        self.assertEquals(len(tags_xml), 4)
        self.assertEquals(tags_xml.tag, 'tags') 
        self.assertEquals(tags_xml[3].tag, 'tag')
        self.assertEquals(tags_xml[3][0].tag, 'name') 
        self.assertEquals(tags_xml[3][1].tag, 'revision') 
        self.assertEquals(tags_xml[3][1].text, '3') 

